<?php

namespace Drupal\unsdg_module\Controllers\Backoffice\ExcelUploads;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\unsdg_module\Services\ExcelServices\Interfaces\ExcelDataServiceInterface;
use Drupal\unsdg_module\Services\ExcelServices\ExcelService;
use Drupal\unsdg_module\Services\SdgDataPointsServices\DataPointsService;
use Symfony\Component\HttpFoundation\File\UploadedFile;

class ExcelUploadController extends ControllerBase
{
    protected $excelService;
    protected $dataPointsService;

    public function __construct(ExcelService $excelService, DataPointsService $dataPointsService)
    {
        $this->excelService = $excelService;
        $this->dataPointsService = $dataPointsService;
    }
    
    //endpoint: /api/excel/indices-target
    public function importIndicesTargetExcel(Request $request)
    {
        $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
        if (!$logged_in) {
            return new JsonResponse("You are not authorized to update any SDG data.", 401);
        }

        try {   
            if (!$request->files->has('file')) {
                return new JsonResponse(['error' => 'File not uploaded'], 400);
            }
            $uploadedFile = $request->files->get('file');
            $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
            $filename = $uploadedFile->getClientOriginalName(); // Use the original name of the file

            $substring = "Indices_Target";

            if (strpos($filename, $substring) === false) {
                return new JsonResponse(['error' => 'Unexpected file, was expecting Indices_Target excel'], 417);
            }
            $filePath = $publicFilesDirectory . '/' . $filename;
            $uploadedFile->move($publicFilesDirectory, $filename);

            if (!file_exists($filePath)) {
                return new JsonResponse(['error' => 'File not found'], 404);
            }
            // $success = $this->excelService->importIndicesTargetExcel($filePath);
            $success = $this->excelService->importIndicesTargetSpreadsheet($filePath);
            

            if ($success) {
                unlink($filePath);
                return new JsonResponse([
                    'message' => 'Excel data imported successfully',
                    'response' => $success
                ], 200);
            } else {
                return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
            }
        } catch (\Exception $e) {
            \Drupal::logger('Import Indices Target Controller')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
            return new JsonResponse(['error' => 'Internal server error'], 500);
        }
    }

    //endpoint: /api/excel/indices-indicator
    public function importIndicesIndicatorExcel(Request $request)
    {
        $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
        if (!$logged_in) {
            return new JsonResponse("You are not authorized to update any SDG data.", 401);
        }

        try {

            ignore_user_abort(true);
            set_time_limit(0);

            if (!$request->files->has('file')) {
                return new JsonResponse(['error' => 'File not uploaded'], 400);
            }
            $uploadedFile = $request->files->get('file');
            $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
            $filename = $uploadedFile->getClientOriginalName(); // Use the original name of the file

            $substring = "Indices_Indicator";

            if (strpos($filename, $substring) === false) {
                return new JsonResponse(['error' => 'Unexpected file, was expecting Indices_Indicator excel'], 417);
            }
            $filePath = $publicFilesDirectory . '/' . $filename;
            $uploadedFile->move($publicFilesDirectory, $filename);

            if (!file_exists($filePath)) {
                return new JsonResponse(['error' => 'File not found'], 404);
            }

            $success1 = $this->excelService->importIndicesUniqueIndicatorSpreadhseet($filePath);
            $success2 = $this->excelService->importIndicesIndicatorSpreadsheet($filePath);

            // Both functions need to succeed for $success to be true
            $success = $success1 && $success2;

            if ($success) {
                unlink($filePath); // Delete the file if both imports are successful
                return new JsonResponse([
                    'message' => 'Excel data imported successfully',
                    'response' => $success
                ], 200);
            } else {
                return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
            }

        } catch (\Exception $e) {
            \Drupal::logger('your_module')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
            return new JsonResponse(['error' => 'Internal server error'], 500);
        }
    }

    //endpoint: /api/excel/indices-goal
    public function importIndicesGoalExcel(Request $request)
    {
        $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
        if (!$logged_in) {
            return new JsonResponse("You are not authorized to update any SDG data.", 401);
        }

        try {
            if (!$request->files->has('file')) {
                return new JsonResponse(['error' => 'File not uploaded'], 400);
            }
            $uploadedFile = $request->files->get('file');
            $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
            $filename = $uploadedFile->getClientOriginalName(); // Use the original name of the file

            $substring = "Indices_Goal"; // typical name is /2.Indices_Goal_test.xlsx

            if (strpos($filename, $substring) === false) {
                return new JsonResponse(['error' => 'Unexpected file, was expecting Indices_Goal excel'], 417);
            }
            $filePath = $publicFilesDirectory . '/' . $filename;
            $uploadedFile->move($publicFilesDirectory, $filename);

            if (!file_exists($filePath)) {
                return new JsonResponse(['error' => 'File not found'], 404);
            }
            $success = $this->excelService->importIndicesGoalSpreadsheet($filePath);

            if ($success) {
                unlink($filePath);
                return new JsonResponse([
                    'message' => 'Excel data imported successfully',
                    'response' => $success
                ], 200);
            } else {
                return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
            }
        } catch (\Exception $e) {
            \Drupal::logger('your_module')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
            return new JsonResponse(['error' => 'Internal server error'], 500);
        }
    }

    //endpoint: /api/excel/sdg-performers
    public function importSDGPerformersExcel(Request $request)
    {
        $logger = \Drupal::logger('sdg_performers_import');
        
        // Check authentication
        $logged_in = \Drupal::currentUser()->isAuthenticated();
        if (!$logged_in) {
            $logger->warning('Unauthorized access attempt detected');
            return new JsonResponse("You are not authorized to update any SDG data.", 401);
        }

        try {
            // Debug request information
            $logger->notice('File upload attempt. Request details: @details', [
                '@details' => json_encode([
                    'files_count' => $request->files->count(),
                    'post_max_size' => ini_get('post_max_size'),
                    'upload_max_filesize' => ini_get('upload_max_filesize'),
                    'content_length' => $request->headers->get('content-length'),
                    'content_type' => $request->headers->get('content-type')
                ])
            ]);

            if (!$request->files->has('file')) {
                $logger->error('No file found in request');
                return new JsonResponse([
                    'error' => 'File not uploaded',
                    'debug' => [
                        'files_present' => array_keys($request->files->all()),
                        'post_data' => array_keys($request->request->all())
                    ]
                ], 400);
            }

            $uploadedFile = $request->files->get('file');
            
            // Validate uploaded file
            if (!$uploadedFile || !$uploadedFile->isValid()) {
                $error = $uploadedFile ? $uploadedFile->getError() : 'Invalid file upload';
                $logger->error('File upload validation failed: @error', ['@error' => $error]);
                return new JsonResponse(['error' => 'Invalid file upload', 'details' => $error], 400);
            }

            // Get and validate file system directory
            $fileSystem = \Drupal::service('file_system');
            $publicFilesDirectory = $fileSystem->realpath("public://");
            
            if (!$publicFilesDirectory || !is_writable($publicFilesDirectory)) {
                $logger->error('Public files directory not writable: @path', ['@path' => $publicFilesDirectory]);
                return new JsonResponse(['error' => 'Server configuration error'], 500);
            }

            // Log directory permissions
            $perms = fileperms($publicFilesDirectory);
            $ownerInfo = posix_getpwuid(fileowner($publicFilesDirectory));
            $groupInfo = posix_getgrgid(filegroup($publicFilesDirectory));
            
            $logger->notice('Directory information: @info', [
                '@info' => json_encode([
                    'path' => $publicFilesDirectory,
                    'permissions' => substr(sprintf('%o', $perms), -4),
                    'owner' => $ownerInfo['name'],
                    'group' => $groupInfo['name']
                ])
            ]);

            $filename = $uploadedFile->getClientOriginalName();
            $substring = "SDG-Performers";
            $summarySubstring = "SDG-Performers-Summary";

            // Validate filename
            if (strpos($filename, $substring) === false) {
                $logger->warning('Invalid file name attempted: @filename', ['@filename' => $filename]);
                return new JsonResponse(['error' => 'Unexpected file, was expecting SDG-Performers excel'], 417);
            }

            if (strpos($filename, $summarySubstring) !== false) {
                $logger->warning('Summary file uploaded instead of main file: @filename', ['@filename' => $filename]);
                return new JsonResponse(['error' => 'Unexpected file, was expecting SDG-Performers excel'], 417);
            }

            $filePath = $publicFilesDirectory . '/' . $filename;

            // Handle existing file
            if (file_exists($filePath)) {
                $logger->notice('Removing existing file: @path', ['@path' => $filePath]);
                unlink($filePath);
            }

            try {
                $uploadedFile->move($publicFilesDirectory, $filename);
            } catch (\Exception $e) {
                $logger->error('File move failed: @error', ['@error' => $e->getMessage()]);
                return new JsonResponse(['error' => 'Failed to save uploaded file'], 500);
            }

            if (!file_exists($filePath)) {
                $logger->error('File not found after move: @path', ['@path' => $filePath]);
                return new JsonResponse(['error' => 'File not found after upload'], 404);
            }

            // Log file permissions after upload
            $filePerms = fileperms($filePath);
            $fileOwner = posix_getpwuid(fileowner($filePath));
            $fileGroup = posix_getgrgid(filegroup($filePath));
            
            $logger->notice('Uploaded file information: @info', [
                '@info' => json_encode([
                    'path' => $filePath,
                    'permissions' => substr(sprintf('%o', $filePerms), -4),
                    'owner' => $fileOwner['name'],
                    'group' => $fileGroup['name'],
                    'size' => filesize($filePath)
                ])
            ]);

            // Import the file
            $success = $this->excelService->importSDGPerformersSpreadsheet($filePath);

            if ($success) {
                // Clean up the file
                if (file_exists($filePath)) {
                    unlink($filePath);
                    $logger->notice('File successfully processed and removed: @filename', ['@filename' => $filename]);
                }
                return new JsonResponse([
                    'message' => 'Excel data imported successfully',
                    'response' => $success
                ], 200);
            } else {
                $logger->error('Excel import failed for file: @filename', ['@filename' => $filename]);
                return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
            }

        } catch (\Exception $e) {
            $logger->error('Error handling Excel upload: @error', [
                '@error' => $e->getMessage(),
                '@trace' => $e->getTraceAsString()
            ]);
            return new JsonResponse([
                'error' => 'Internal server error',
                'message' => $e->getMessage()
            ], 500);
        }
    }
    // public function importSDGPerformersExcel(Request $request)
    // {
    //     $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
    //     if (!$logged_in) {
    //         return new JsonResponse("You are not authorized to update any SDG data.", 401);
    //     }

    //     try {
    //         if (!$request->files->has('file')) {
    //             return new JsonResponse(['error' => 'File not uploaded'], 400);
    //         }
    //         $uploadedFile = $request->files->get('file');
    //         $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
    //         $filename = $uploadedFile->getClientOriginalName(); // Use the original name of the file

    //         $substring = "SDG-Performers"; // typical name is SDG-Performers_2024-06-25.xlsx
    //         $summarySubstring = "SDG-Performers-Summary";// perform a second check because of potential substring of the name of another excel
    //         if (strpos($filename, $substring) === false) {
    //             return new JsonResponse(['error' => 'Unexpected file, was expecting SDG-Performers excel'], 417);
    //         }
    //         if (strpos($filename, $summarySubstring) === true) {
    //             return new JsonResponse(['error' => 'Unexpected file, was expecting SDG-Performers excel'], 417);
    //         }

    //         $filePath = $publicFilesDirectory . '/' . $filename;
    //         $uploadedFile->move($publicFilesDirectory, $filename);

    //         if (!file_exists($filePath)) {
    //             return new JsonResponse(['error' => 'File not found'], 404);
    //         }
    //         // $success = $this->excelService->importSDGPerformersExcel($filePath);
    //         $success = $this->excelService->importSDGPerformersTestExcel($filePath);


    //         if ($success) {
    //             unlink($filePath);
    //             return new JsonResponse([
    //                 'message' => 'Excel data imported successfully',
    //                 'response' => $success
    //             ], 200);
    //         } else {
    //             return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
    //         }
    //     } catch (\Exception $e) {
    //         \Drupal::logger('your_module')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
    //         return new JsonResponse(['error' => 'Internal server error'], 500);
    //     }
    // }

    //endpoint: /api/excel/sdg-performers-summary
    public function importSDGPerformersSummaryExcel(Request $request)
    {
        $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
        if (!$logged_in) {
            return new JsonResponse("You are not authorized to update any SDG data.", 401);
        }

        try {
            if (!$request->files->has('file')) {
                return new JsonResponse(['error' => 'File not uploaded'], 400);
            }
            $uploadedFile = $request->files->get('file');
            $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
            $filename = $uploadedFile->getClientOriginalName(); // Use the original name of the file

            $substring = "SDG-Performers-Summary";// typical name is SDG-Performers-Summary_2024-06-25.xlsx

            if (strpos($filename, $substring) === false) {
                return new JsonResponse(['error' => 'Unexpected file, was expecting SDG-Performers-Summary excel'], 417);
            }

            $filePath = $publicFilesDirectory . '/' . $filename;
            $uploadedFile->move($publicFilesDirectory, $filename);

            if (!file_exists($filePath)) {
                return new JsonResponse(['error' => 'File not found'], 404);
            }
            $success = $this->excelService->importSDGPerformersSummarySpreadsheet($filePath);

            if ($success) {
                unlink($filePath);
                return new JsonResponse([
                    'message' => 'Excel data imported successfully',
                    'response' => $success
                ], 200);
            } else {
                return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
            }
        } catch (\Exception $e) {
            \Drupal::logger('Import SDG Performers Summary Spreadsheet')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
            return new JsonResponse(['error' => 'Internal server error'], 500);
        }
    }

    //endpoint: /api/excel/sdg-data-availability
    public function importDataAvailabilityExcel(Request $request)
    {
        $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
        if (!$logged_in) {
            return new JsonResponse("You are not authorized to update any SDG data.", 401);
        }

        try {
            if (!$request->files->has('file')) {
                return new JsonResponse(['error' => 'File not uploaded'], 400);
            }
            $uploadedFile = $request->files->get('file');
            $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
            $filename = $uploadedFile->getClientOriginalName(); // Use the original name of the file

            $substring = "DataAvailability-Indicator";// typical name is 3a.DataAvailability-Indicator-2023.xlsx

            if (strpos($filename, $substring) === false) {
                return new JsonResponse(['error' => 'Unexpected file, was expecting DataAvailability-Indicator excel'], 417);
            }

            $filePath = $publicFilesDirectory . '/' . $filename;
            $uploadedFile->move($publicFilesDirectory, $filename);

            if (!file_exists($filePath)) {
                return new JsonResponse(['error' => 'File not found'], 404);
            }
            // $success = $this->excelService->importSDGDataAvailabilityExcel($filePath);
            $success = $this->excelService->importDataAvailabilitySpreadsheet($filePath);

            if ($success) {
                $isUpdateSuccess = $this->dataPointsService->calculateRegionalDataAvailabilityByRegionCountryGroups();
                unlink($filePath);
                return new JsonResponse([
                    'message' => 'Excel data imported successfully',
                    'response' => $success
                ], 200);
            } else {
                return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
            }
        } catch (\Exception $e) {
            \Drupal::logger('your_module')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
            return new JsonResponse(['error' => 'Internal server error'], 500);
        }
    }

    //endpoint: /api/excel/escap-data-sources
    public function importESCAPDataSources(Request $request)
    {
        $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
        if (!$logged_in) {
            return new JsonResponse("You are not authorized to update any SDG data.", 401);
        }

        try{
            if (!$request->files->has('file')) {
                return new JsonResponse(['error' => 'File not uploaded'], 400);
            }
            $uploadedFile = $request->files->get('file');
            $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
            $filename = $uploadedFile->getClientOriginalName(); // Use the original name of the file

            if($filename !== 'ESCAP-SDG-DataSources.xlsx'){
                // ESCAP-SDG-DataSources.xlsx
                return new JsonResponse(['error' => 'Unexpected file, was expecting ESCAP-SDG-DataSources.xlsx'], 417);
            }
            $filePath = $publicFilesDirectory . '/' . $filename;
            $uploadedFile->move($publicFilesDirectory, $filename);

            if (!file_exists($filePath)) {
                return new JsonResponse(['error' => 'File not found'], 404);
            }
            $success = $this->excelService->importSDGDataSourcesExcel($filePath);

            if ($success) {
                unlink($filePath);
                return new JsonResponse([
                    'message' => 'Excel data imported successfully',
                    'response' => $success
                ], 200);
            } else {
                return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
            }
        }
        catch (\Exception $e) {
            \Drupal::logger('your_module')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
            return new JsonResponse(['error' => 'Internal server error'], 500);
        }
    }

    // endpoint: /api/excel/goal-data
    // public function importCountryGoalSnapshotData(Request $request)
    // {
    //     $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
    //     if (!$logged_in) {
    //         return new JsonResponse("You are not authorized to update any SDG data.", 401);
    //     }

    //     try {
    //         if (!$request->files->has('file')) {
    //             return new JsonResponse(['error' => 'File not uploaded'], 400);
    //         }
    //         $uploadedFile = $request->files->get('file');
    //         $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
    //         $filename = $uploadedFile->getClientOriginalName();

    //         $filePath = $publicFilesDirectory . '/' . $filename;

    //         $uploadedFile->move($publicFilesDirectory, $filename);

    //         $parts = explode(' ', ltrim($filename, '/'));
    //         $countryCode = $parts[0];
    //         $yearWithExtension = end($parts);
    //         $year = pathinfo($yearWithExtension, PATHINFO_FILENAME);

    //         if($countryCode === "PHL" || $countryCode === "MNG"){
    //             $success = $this->excelService->importCountryGoalSnapshotDataIrregularExcel($filePath, $countryCode, $year);

    //         }
    //         else{
    //             $success = $this->excelService->importCountryGoalSnapshotDataExcel($filePath, $countryCode, $year);
    //         }


    //         if ($success) {
    //             unlink($filePath);
    //         }


    //         if ($success) {
    //             return new JsonResponse([
    //                 'message' => 'Excel data imported successfully',
    //                 'response' => $success
    //             ], 200);
    //         } else {
    //             return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
    //         }
    //     } catch (\Exception $e) {

    //         \Drupal::logger('your_module')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
    //         return new JsonResponse(['error' => 'Internal server error'], 500);
    //     }
    // }

    //endpoint: /api/excel/goal-data
    public function importCountryGoalSnapshotData(Request $request)
    {
        // $logged_in = \Drupal::currentUser()->isAuthenticated(); //check if backoffice session exists
        // if (!$logged_in) {
        //     return new JsonResponse("You are not authorized to update any SDG data.", 401);
        // }

        try {
            // Check if file was uploaded
            if (!$request->files->has('file')) {
                return new JsonResponse(['error' => 'File not uploaded'], 400);
            }

            // Get query parameters
            $countryCode = $request->query->get('country_code');
            $year = $request->query->get('year');

            // Validate country code
            if (!in_array($countryCode, ['MNG', 'LKA', 'TLS', 'TJK', 'FJI', 'BRN', 'PHL'])) {
                return new JsonResponse(['error' => 'Invalid country code'], 400);
            }

            $uploadedFile = $request->files->get('file');
            $filename = $uploadedFile->getClientOriginalName();

            // Validate file name format and match with query parameters
            $expectedPattern = "/^SDG-{$countryCode}-{$year}\.csv$/";
            if (!preg_match($expectedPattern, $filename)) {
                return new JsonResponse([
                    'error' => 'Invalid file name format. Expected format: SDG-[COUNTRY_CODE]-[YEAR].csv',
                    'details' => "File should be named: SDG-{$countryCode}-{$year}.csv"
                ], 400);
            }

            // Process the validated file
            $publicFilesDirectory = \Drupal::service('file_system')->realpath("public://");
            $filePath = $publicFilesDirectory . '/' . $filename;
            
            // Move the file
            try {
                $uploadedFile->move($publicFilesDirectory, $filename);
            } catch (\Exception $e) {
                return new JsonResponse(['error' => 'Failed to save uploaded file'], 500);
            }

            // Import the data
            $success = $this->excelService->importGoalSnapshot($filePath, $year, $countryCode);
            
            // Clean up the file regardless of import success
            if (file_exists($filePath)) {
                unlink($filePath);
            }

            if ($success) {
                return new JsonResponse([
                    'message' => 'Excel data imported successfully',
                    'response' => $success
                ], 200);
            } else {
                return new JsonResponse(['error' => 'Failed to import Excel data'], 500);
            }
        } catch (\Exception $e) {
            \Drupal::logger('your_module')->error('Error handling Excel upload: @error', ['@error' => $e->getMessage()]);
            return new JsonResponse(['error' => 'Internal server error'], 500);
        }
    }

}
