<?php

namespace Drupal\unsdg_module\Services\ContentServices;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryFactoryInterface;
use Drupal\unsdg_module\Models\Country;
use Drupal\unsdg_module\Models\CountryGroup;
use Drupal\node\NodeInterface;
use Drupal\node\Entity\Node;


class RegionService
{
    protected $entityTypeManager;

    public function __construct(EntityTypeManagerInterface $entityTypeManager,)
    {
        $this->entityTypeManager = $entityTypeManager;
    }

    public function getAllRegions()
    {
        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->accessCheck(TRUE);
        $query->condition('type', 'region');
        $nids = $query->execute();
        $regions = [];
        foreach ($nids as $nid) {
            $node = $this->entityTypeManager->getStorage('node')->load($nid);
            if ($node) {
                $countryGroups = $this->getRegionCountryGroups($node);
                $regions[] = [
                    'id' => $node->id(),
                    'name' => $node->label(),
                    'countryGroups' => $countryGroups
                ];
            }
        }
        return $regions;
    }

    public function getRegionByName($regionName)
    {
        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->accessCheck(TRUE);
        $query->condition('type', 'region')
            ->condition('title', $regionName)->range(0, 1);
        $nids = $query->execute();
        $region = null;
        foreach ($nids as $nid) {
            $region = $this->entityTypeManager->getStorage('node')->load($nid);
            if ($region) {
                $countryGroups = $this->getRegionCountryGroups($region);
                $regionData[] = [
                    'id' => $region->id(),
                    'name' => $region->label(),
                    'countryGroups' => $countryGroups,
                ];
            }
            break;
        }
        return $regionData;
    }

    public function getCountryGroupAbbreviationByCountryGroupName($countryGroupName)
    {
        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->accessCheck(TRUE);
        $query->condition('type','country_group')
        ->condition('title', $countryGroupName)->range(0,1);

        $nids = $query->execute();
        foreach ($nids as $nid) {
            $countryGroup = $this->entityTypeManager->getStorage('node')->load($nid);
            $isoCode = $this->getIsoCodeValue($countryGroup);
            break;
        }

        return $isoCode;
    }

    public function getIsoCodeValue($node)
    {
        $isoCodeValue = null;

        if ($node->hasField('field_iso_code') && !$node->get('field_iso_code')->isEmpty()) {
            $field_iso_code = $node->get('field_iso_code')->getValue();
            $isoCodeValue = $field_iso_code[0]['value'];
        }

        return $isoCodeValue;
    }

    public function getCountriesFromCountryGroupByCountryName($countryGroupName)
    {
        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->accessCheck(TRUE);
        $query->condition('type', 'country_group')
        ->condition('title', $countryGroupName)->range(0, 1); // Optimize query to fetch only one result
        $nids = $query->execute();
        foreach ($nids as $nid) {
            $countryGroup = $this->entityTypeManager->getStorage('node')->load($nid);
            $countriesDetails = $this->getCountriesDetailsFromCountryGroup($countryGroup);
            break;
        }
        return $countriesDetails;
    }

    public function getCountriesIsoFromCountryGroupByCountryName($countryGroupName)
    {
        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->accessCheck(TRUE);
        $query->condition('type', 'country_group')
        ->condition('title', $countryGroupName)->range(0, 1);
        $nids = $query->execute();
        foreach ($nids as $nid) {
            $countryGroup = $this->entityTypeManager->getStorage('node')->load($nid);
            $countryIsoCodes = $this->getCountriesIsoFromCountryGroup($countryGroup);
            break;
        }
        return $countryIsoCodes;
    }

    public function getCountriesDetailsFromCountryGroup($countryGroup)
    {
        $countriesField = $countryGroup->get('field_countries')->referencedEntities();
        foreach ($countriesField as $country) {
            $countryList[] = [
                'id' => $country->id(),
                'name' => $country->label(),
                'isoCode' => $country->get('field_country_abbreviation')->value,
            ];
        }
        return $countryList;
    }

    public function getCountriesIsoFromCountryGroupName($countryGroupName)
    {
        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->accessCheck(TRUE);
        $query->condition('type', 'country_group')
        ->condition('title', $countryGroupName)->range(0, 1);
        $nids = $query->execute();
        foreach ($nids as $nid) {
            $countryGroup = $this->entityTypeManager->getStorage('node')->load($nid);
            $countryIsoCodes = $this->getCountriesIsoFromCountryGroup($countryGroup);
        }
        return $countryIsoCodes;
    }

    public function getCountriesIsoFromCountryGroup($countryGroup)
    {
        $countriesField = $countryGroup->get('field_countries')->referencedEntities();
        foreach ($countriesField as $country) {
            $countryIsoCode[] = $country->get('field_country_abbreviation')->value;
        }
        return $countryIsoCode;
    }

    public function getRegionCountryGroups($node)
    {
        $countryGroups = [];
        if ($node->hasField('field_country_groups') && !$node->get('field_country_groups')->isEmpty()) {
            $countryGroupsField = $node->get('field_country_groups')->referencedEntities();
            foreach ($countryGroupsField as $countryGroup) {
                $countryList = [];
                $countriesField = $countryGroup->get('field_countries')->referencedEntities();
                foreach ($countriesField as $country) {
                    $countryList[] = [
                        'id' => $country->id(),
                        'name' => $country->label(),
                        'isoCode' => $country->get('field_country_abbreviation')->value,
                    ];
                }
                $countryGroups[] = [
                    'id' => $countryGroup->id(),
                    'name' => $countryGroup->label(),
                    'isoCode' => $countryGroup->get('field_iso_code')->value,
                    'countries' => $countryList,
                ];
            }
        }
        return $countryGroups;
    }

    // public function getReferencedCountryGroups($node)
    // {
    //     $countries = [];
    //     if ($node->hasField('field_country_groups') && !$node->get('field_country_groups')->isEmpty()) {
    //         $country_references = $node->get('field_country_groups')->referencedEntities();
    //         foreach ($country_references as $country) {
    //             $countries[] = new Country(
    //                 $country->id(),
    //                 $country->label(),
    //                 $country->get('field_country_abbreviation')->value);
    //         }
    //     }
    //     return $countries;
    // }

    public function getCountryGroupFromCountry($countryIsoCode)
    {
        $regions = $this->getRegionByName('Region');
        foreach($regions as $region){
            foreach ($region['countryGroups'] as $countryGroup) {
                foreach ($countryGroup['countries'] as $country) {
                    if ($country['isoCode'] === $countryIsoCode) {
                        return $countryGroup;
                    }
                }
            }
        }
        return null;
    }


    public function getCountryGroupCountries($node)
    {
        $countries = [];
        if ($node->hasField('field_countries') && !$node->get('field_countries')->isEmpty()) {
            $countryReferences = $node->get('field_countries')->referencedEntities();
            foreach ($countryReferences as $country) {
                $countries[] = [
                    'id' => $country->id(),
                    'name' => $country->label(),
                    'isoCode' => $country->get('field_country_abbreviation')->value,
                ];
            }
        }
        return $countries;
    }

    public function getReferencedCountriesAbbreviations($node)
    {
        $countriesAbbreviations = [];
        if ($node->hasField('field_countries') && !$node->get('field_countries')->isEmpty()) {
            $country_references = $node->get('field_countries')->referencedEntities();
            foreach ($country_references as $country) {
                $countriesAbbreviations[] = $country->get('field_country_abbreviation')->value;
            }
        }
        return $countriesAbbreviations;
    }

    public function getCountries()
    {
        $query = \Drupal::entityQuery('node');
        $query->accessCheck(FALSE)
            ->condition('type', 'country');
        $nids = $query->execute();
        $countries = [];
        if ($nids) {
            foreach ($nids as $nid) {
                $node = Node::load($nid);
                $countries[] = [
                    'id' => $node->id(),
                    'name' => $node->label(),
                    'isoCode' => $node->get('field_country_abbreviation')->value,
                ];
            }
        }
        return $countries;
    }
}
