<?php

namespace Drupal\unsdg_module\Services\ContentServices;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\taxonomy\Entity\Term;
use Drupal\node\Entity\Node;
use Drupal\Core\Url;
use Drupal\unsdg_module\Models\PaginatedList;

class ResourceService
{
    protected $entityTypeManager;

    public function __construct(EntityTypeManagerInterface $entityTypeManager)
    {
        $this->entityTypeManager = $entityTypeManager;
    }

    public function getPaginatedResourceNodes($taxonomyTermId, $pageNumber, $pageSize): PaginatedList
    {
        $nodeStorage = $this->entityTypeManager->getStorage('node');
        $query = $nodeStorage
            ->getQuery()
            ->condition('type', 'resource')
            ->accessCheck(true);

        if ($taxonomyTermId) {
            $term = Term::load($taxonomyTermId);

            if ($term) {
                $query->condition('field_tags', $taxonomyTermId);
            }
        }

        $countQuery = clone $query;
        $count = $countQuery->count()->execute();

        $offset = ($pageNumber - 1) * $pageSize;
        $query->range($offset, $pageSize);
        $nodeIds = $query->execute();

        $nodes = [];
        foreach ($nodeIds as $nid) {
            $node = $nodeStorage->load($nid);
            if (
                isset($node->field_image)
                && $node->field_image->isEmpty() == false
            ) {
                $file_entity = $node->field_image->entity;

                if ($file_entity) {
                    $file_uri = $node->field_image->entity->uri->value;
                    $url = \Drupal::service('file_url_generator')->generateString($file_uri);
                    $node->image_url = $url;
                }
            }
            $url_object = Url::fromRoute('entity.node.canonical', ['node' => $node->id()]);
            $url = $url_object->toString();
            $node->url = $url;
            $nodes[] = $node;
        }

        foreach ($nodes as $resource) {
            $tags = [];
            if (isset($resource->field_tags->target_id)) {
                $tag_ids = (array) $resource->field_tags->target_id;
                $tagEntities = \Drupal\taxonomy\Entity\Term::loadMultiple($tag_ids);
                foreach ($tagEntities as $term) {
                    $tag = [
                        'id' => $term->id(),
                        'name' => $term->getName(),
                    ];
                    $tags[] = $tag;
                }
            }

            $resources_data[] = [
                'id' => $resource->id(),
                'title' => $resource->label(),
                'content' => $resource->field_resource_content->value,
                'tags' => $tags,
                'image' => $resource->image_url,
                'url' => $resource->url,
            ];
        }

        return new PaginatedList($resources_data, $pageNumber, $pageSize, $count);
    }

    public function getResourcesFromResourceNodes($resourceNodes) {}

    public function getRelatedResourcesFromResourceNodes($resourceNodes, $resourceId)
    {
        $resources_data = [];
        foreach ($resourceNodes as $resource) {
            $tags = [];
            if (isset($resource->field_tags->target_id)) {
                
                // skip the resource if its id matches $resourceId
                if ($resource->id() == $resourceId) {
                    continue;
                }

                $tag_ids = (array) $resource->field_tags->target_id;
                $tagEntities = \Drupal\taxonomy\Entity\Term::loadMultiple($tag_ids);
                foreach ($tagEntities as $term) {
                    $tag = [
                        'id' => $term->id(),
                        'name' => $term->getName(),
                    ];
                    $tags[] = $tag;
                }
            }

            $resources_data[] = [
                'id' => $resource->id(),
                'title' => $resource->label(),
                //'content' => $resource->field_resource_content->value,
                'tags' => $tags,
                'image' => $resource->image_url,
                'url' => $resource->url,
            ];
        }
        return $resources_data;
    }

    public function getRelatedResourceNodes($resourceId, $relatedCount = 3)
    {
        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->condition('type', 'resource')
            ->sort('created', 'DESC')
            ->range(0, $relatedCount);
        $query->accessCheck(FALSE);

        $resource = $this->getResourceById($resourceId);
        $taxonomyTermId = $resource[0]['tags'][0]['id'];
        if ($taxonomyTermId) {
            $term = Term::load($taxonomyTermId);
            if ($term) {
                $query->condition('field_tags', $taxonomyTermId);
            }
        }
        $nids = $query->execute();
        $nodes = [];
        foreach ($nids as $nid) {
            $node = Node::load($nid);
            if ($node->hasField('field_image') && !$node->get('field_image')->isEmpty()) {
                $file_entity = $node->field_image->entity;
                if ($file_entity) {
                    $file_uri = $node->field_image->entity->uri->value;
                    $url = \Drupal::service('file_url_generator')->generateString($file_uri);
                    $node->image_url = $url;
                }
            }
            $url_object = Url::fromRoute('entity.node.canonical', ['node' => $node->id()]);
            $url = $url_object->toString();
            $node->url = $url;
            $nodes[] = $node;
        }

        return $nodes;
    }

    public function getResourceById($resourceId)
    {
        $resource = Node::load($resourceId);
        if ($resource->hasField('field_image') && !$resource->get('field_image')->isEmpty()) {
            $file_entity = $resource->field_image->entity;
            if ($file_entity) {
                $file_uri = $resource->field_image->entity->uri->value;
                $url = \Drupal::service('file_url_generator')->generateString($file_uri);
                $resource->image_url = $url;
            }
        }
        $url_object = Url::fromRoute('entity.node.canonical', ['node' => $resource->id()]);
        $url = $url_object->toString();
        $resource->url = $url;

        $tags = [];
        if (isset($resource->field_tags->target_id)) {
            $tag_ids = (array) $resource->field_tags->target_id;
            $tagEntities = \Drupal\taxonomy\Entity\Term::loadMultiple($tag_ids);
            foreach ($tagEntities as $term) {
                $tag = [
                    'id' => $term->id(),
                    'name' => $term->getName(),
                ];
                $tags[] = $tag;
            }
        }

        $resourceResult[] = [
            'id' => $resource->id(),
            'title' => $resource->label(),
            'content' => $resource->field_resource_content->value,
            'tags' => $tags,
            'image' => $resource->image_url,
            'url' => $resource->url,
        ];
        return $resourceResult;
    }

    public function getAllTagsInVocabulary($vocabularyMachineName)
    {
        //get vocabularyID from machine name
        $vocabulary = \Drupal\taxonomy\Entity\Vocabulary::load($vocabularyMachineName);
        if (!$vocabulary) {
            return [];
        }

        //entity query for taxonomy terms
        $query = \Drupal::entityQuery('taxonomy_term');
        $query->condition('vid', $vocabulary->id());
        $query->accessCheck(TRUE);

        $tids = $query->execute();
        $tags = [];
        $terms = \Drupal\taxonomy\Entity\Term::loadMultiple($tids);
        foreach ($terms as $term) {
            $tag = [
                'id' => $term->id(),
                'name' => $term->getName(),
            ];
            $tags[] = $tag;
        }

        return $tags;
    }

    public function getResourcesTags()
    {
        $tags = $this->getAllTagsInVocabulary('resource');
        return $tags;
    }
}
