<?php

namespace Drupal\unsdg_module\Services\ExcelServices;

use Drupal\Component\Uuid\Uuid;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Drupal\Core\Database\Connection;
use Drupal\unsdg_module\Services\ExcelServices\Interfaces\ExcelDataServiceInterface;
use Drupal\unsdg_module\Services\ExcelServices\Interfaces\ChunkReadFilter;
use Drupal\Core\Database\Database;
use Drupal\Core\Entity\Query\QueryFactoryInterface;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;

// // Assuming 'custom' database settings are defined in settings.ddev.php.
// $database_settings = $databases['custom']['default'];

// // Extract connection parameters.
// $database_name = $database_settings['database'];
// $database_username = $database_settings['username'];
// $database_password = $database_settings['password'];
// $database_host = $database_settings['host'];
// $database_port = $database_settings['port'];
// $database_driver = $database_settings['driver'];

class ExcelService implements ExcelDataServiceInterface
{
    protected $queryFactory;
    public function __construct(QueryFactoryInterface $queryFactory)
    {
        $this->queryFactory = $queryFactory;
    }

    private function normalizeSnapshotGoal($value)
    {
        $trimmedValue = trim($value);
        if ($trimmedValue === '' || $trimmedValue === '0') {
            return (float) 0;
        } elseif (is_numeric($trimmedValue)) {
            return (float) $trimmedValue;
        } else {
            return null;
        }
    }

    private function executeBatchInsert($database, $table, $columns, $values, $placeholders)
    {
        $query = "INSERT INTO $table (" . 
            implode(', ', $columns) .
            ") VALUES " . implode(', ', $placeholders);
        
        $database->query($query, $values);
    }

    /**
     * Sanitize a value for database insertion
     */
    private function sanitizeValue($value, $column = null) {
        // Convert null or empty values to NULL
        if ($value === null || $value === '') {
            return null;
        }
        
        // Convert boolean values
        if (is_bool($value)) {
            return $value ? 1 : 0;
        }
        
        // Handle numeric values
        if (is_numeric($value)) {
            return $value;
        }
        
        // For strings, remove any potentially problematic characters
        if (is_string($value)) {
            // Remove any null bytes
            $value = str_replace("\0", '', $value);
            
            // Replace multiple spaces with single space
            $value = preg_replace('/\s+/', ' ', $value);
            
            // Trim whitespace
            $value = trim($value);
            
            return $value;
        }
        
        return $value;
    }
    
    private function sanitizeValueGoalSnapshot($value, $column = null)
    {
        // Remove any hidden characters and trim
        $value = trim(preg_replace('/[\x00-\x1F\x7F]/u', '', $value));
        
        // Debug logging for empty values
        if ($value === '' || $value === null) {
            \Drupal::logger('csv_import')->notice("Empty value detected for column: $column");
        }
        
        // Handle empty or null values
        if ($value === '' || $value === null) {
            // For numeric columns, return 0 instead of null
            if (in_array($column, ['grey_bar', 'indicator_percentage'])) {
                return 0.000000;
            }
            return null;
        }
        
        // Special handling for numeric columns
        if (in_array($column, ['grey_bar', 'indicator_percentage']) && is_numeric($value)) {
            return number_format((float)$value, 6, '.', '');
        }
        
        return $value;
    }

    private function detectDelimiter($filePath) {
        $handle = fopen($filePath, 'r');
        if ($handle === FALSE) {
            throw new \Exception("Could not open file: $filePath");
        }
    
        // Read the first line
        $firstLine = fgets($handle);
        fclose($handle);
    
        // Count occurrences of potential delimiters
        $commas = substr_count($firstLine, ',');
        $tabs = substr_count($firstLine, "\t");
    
        // Return the more frequently occurring delimiter
        return ($commas > $tabs) ? ',' : "\t";
    }

    public function importIndicesTargetSpreadsheet(string $filePath)
    {
        try {
            $database = Database::getConnection('default', 'custom');
            $database->query("TRUNCATE TABLE target");
            
            // Define column mapping
            $columnMapping = [
                'target_num' => 2,
                'target_short_name' => 11,
                'target_long_name' => 11,
                'goal_num' => 1,
                'target_color' => 14,
                'country_group' => 0,
                'year' => 3,
                'snapshot_target' => 4,
                'dashboard_target' => 5,
                'snapshot_count' => 6,
                'proxy_count' => 7,
                'g_order' => 8,
                'r_order' => 9,
                't_order' => 12,
                'value' => 13,
                '`order`' => 15,
            ];
            
            // Open CSV file
            $handle = fopen($filePath, 'r');
            if ($handle === FALSE) {
                throw new \Exception("Could not open file: $filePath");
            }
            
            // Skip header row
            fgetcsv($handle);
            
            // Prepare batch insert
            $batchSize = 1000;
            $values = [];
            $placeholders = [];
            
            // Read CSV rows
            while (($row = fgetcsv($handle)) !== FALSE) {
                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }
                
                // Map the columns according to our specification
                $mappedData = [];
                foreach ($columnMapping as $field => $columnIndex) {
                    $mappedData[] = $row[$columnIndex] ?? null;
                }
                
                // Add sanitized values to the array
                $values = array_merge($values, array_map([$this, 'sanitizeValue'], $mappedData));
                $placeholders[] = '(' . implode(', ', array_fill(0, count($columnMapping), '?')) . ')';
                
                // Execute batch insert when batch size is reached
                if (count($placeholders) >= $batchSize) {
                    $this->executeBatchInsert($database, 'target', array_keys($columnMapping), $values, $placeholders);
                    $values = [];
                    $placeholders = [];
                }
            }
            
            // Insert remaining rows
            if (!empty($placeholders)) {
                $this->executeBatchInsert($database, 'target', array_keys($columnMapping), $values, $placeholders);
            }
            
            fclose($handle);
            
            return TRUE;
        } catch (\Exception $e) {
            if (isset($handle)) {
                fclose($handle);
            }
            \Drupal::logger('import indices target test')
                ->error('Error importing CSV data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importIndicesUniqueIndicatorSpreadhseet(string $filePath)
    {
        try {
            $database = Database::getConnection('default', 'custom');
            
            // Create temporary table
            $createTempTableQuery = "CREATE TEMPORARY TABLE indices_indicator_numbers (
                id char(36) PRIMARY KEY NOT NULL DEFAULT (uuid()),
                indicator_number varchar(45) DEFAULT NULL,
                indicator_short_name varchar(500) DEFAULT NULL,
                target_number varchar(45) DEFAULT NULL
            )";
            
            $database->query($createTempTableQuery);
            
            // Define column mapping
            $columnMapping = [
                'indicator_number' => 11,
                'indicator_short_name' => 12,
                'target_number' => 1,
            ];
            
            // Open CSV file
            $handle = fopen($filePath, 'r');
            if ($handle === FALSE) {
                throw new \Exception("Could not open file: $filePath");
            }
            
            // Skip header row
            fgetcsv($handle);
            
            // Prepare batch insert
            $batchSize = 1000;
            $values = [];
            $placeholders = [];
            
            // Read CSV rows
            while (($row = fgetcsv($handle)) !== FALSE) {
                // Map the columns according to our specification
                $mappedData = [];
                foreach ($columnMapping as $field => $columnIndex) {
                    $mappedData[] = $row[$columnIndex] ?? null;
                }
                
                // Add sanitized values to the array
                $values = array_merge($values, array_map([$this, 'sanitizeValue'], $mappedData));
                $placeholders[] = '(?, ?, ?)';
                
                // Execute batch insert when batch size is reached
                if (count($placeholders) >= $batchSize) {
                    $this->executeBatchInsert($database, 'indices_indicator_numbers', array_keys($columnMapping), $values, $placeholders);
                    $values = [];
                    $placeholders = [];
                }
            }
            
            // Insert remaining rows
            if (!empty($placeholders)) {
                $this->executeBatchInsert($database, 'indices_indicator_numbers', array_keys($columnMapping), $values, $placeholders);
            }
            
            fclose($handle);
            
            // Process distinct records
            $database->query("TRUNCATE TABLE indicator");
            
            $database->query("INSERT INTO indicator (indicator_number, indicator_short_name, target_number)
                SELECT DISTINCT 
                    indicator_number,
                    MIN(indicator_short_name) as indicator_short_name,
                    MIN(target_number) as target_number
                FROM indices_indicator_numbers 
                GROUP BY indicator_number");
                
            $database->query("DROP TABLE indices_indicator_numbers");
            
            return TRUE;
        } catch (\Exception $e) {
            if (isset($handle)) {
                fclose($handle);
            }
            \Drupal::logger('Import Indices Unique Indicator Test')
                ->error('Error importing CSV data: @error', ['@error' => $e->getMessage()]);
            throw $e;
        }
    }

    public function importIndicesIndicatorSpreadsheet(string $filePath)
    {
        try {
            $database = Database::getConnection('default', 'custom');
            $database->query("TRUNCATE TABLE indicator_data");
            
            // Define column mapping
            $columnMapping = [
                'indicator_id' => 11,
                'snapshot_indicator' => 3,
                'year' => 2,
                'dashboard_indicator' => 4,
                'country_group' => 8,
                'merged_id' => 7,
                'snapshot_count' => 5
            ];
            
            // Open CSV file
            $handle = fopen($filePath, 'r');
            if ($handle === FALSE) {
                throw new \Exception("Could not open file: $filePath");
            }
            
            // Skip header row
            fgetcsv($handle);
            
            // Prepare batch insert
            $batchSize = 1000;
            $values = [];
            $placeholders = [];
            
            // Read CSV rows
            while (($row = fgetcsv($handle)) !== FALSE) {
                // Map the columns according to our specification
                $mappedData = [];
                foreach ($columnMapping as $field => $columnIndex) {
                    $mappedData[] = $row[$columnIndex] ?? null;
                }
                
                // Add sanitized values to the array
                $values = array_merge($values, array_map([$this, 'sanitizeValue'], $mappedData));
                $placeholders[] = '(' . implode(', ', array_fill(0, count($columnMapping), '?')) . ')';
                
                // Execute batch insert when batch size is reached
                if (count($placeholders) >= $batchSize) {
                    $this->executeBatchInsert($database, 'indicator_data', array_keys($columnMapping), $values, $placeholders);
                    $values = [];
                    $placeholders = [];
                }
            }
            
            // Insert remaining rows
            if (!empty($placeholders)) {
                $this->executeBatchInsert($database, 'indicator_data', array_keys($columnMapping), $values, $placeholders);
            }
            
            fclose($handle);
            
            return TRUE;
        } catch (\Exception $e) {
            if (isset($handle)) {
                fclose($handle);
            }
            \Drupal::logger('Import Indices Indicator Test')
                ->error('Error importing CSV data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importIndicesGoalSpreadsheet(string $filePath)
    {
        try {
            $database = Database::getConnection('default', 'custom');
            $database->query("TRUNCATE TABLE goal_data");
            
            $columnMapping = [
                'country_group' => 0,
                'goal_number' => 1,
                'year' => 2,
                'snapshot_goal' => 3,
                'dashboard_goal' => 4,
                'snapshot_count' => 5,
                'indicator_count' => 8,
                'indicator_percentage' => 11
            ];
            
            // Open CSV file
            $handle = fopen($filePath, 'r');
            if ($handle === FALSE) {
                throw new \Exception("Could not open file: $filePath");
            }
            
            // Skip header row
            fgetcsv($handle);
            
            // Prepare batch insert
            $batchSize = 1000;
            $values = [];
            $placeholders = [];
            
            // Read CSV rows
            while (($row = fgetcsv($handle)) !== FALSE) {
                // Map the columns according to our specification
                $mappedData = [];
                foreach ($columnMapping as $field => $columnIndex) {
                    $mappedData[] = $row[$columnIndex] ?? null;
                }
                
                // Add sanitized values to the array
                $values = array_merge($values, array_map([$this, 'sanitizeValue'], $mappedData));
                $placeholders[] = '(' . implode(', ', array_fill(0, count($columnMapping), '?')) . ')';
                
                // Execute batch insert when batch size is reached
                if (count($placeholders) >= $batchSize) {
                    $this->executeBatchInsert($database, 'goal_data', array_keys($columnMapping), $values, $placeholders);
                    $values = [];
                    $placeholders = [];
                }
            }
            
            // Insert remaining rows
            if (!empty($placeholders)) {
                $this->executeBatchInsert($database, 'goal_data', array_keys($columnMapping), $values, $placeholders);
            }
            
            fclose($handle);
            
            return TRUE;
        } catch (\Exception $e) {
            if (isset($handle)) {
                fclose($handle);
            }
            \Drupal::logger('Import Indices Goal Spreadsheet')
                ->error('Error importing CSV data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }
    public function importSDGPerformersSpreadsheet(string $filePath)
    {
        try {
            $database = Database::getConnection('default', 'custom');
            $database->query("TRUNCATE TABLE sdg_performers_data");

            $columnMapping = [
                'ind_number' => 0,
                'ind_short_name' => 1,
                'goal_number' => 2,
                'target_number' => 3,
                'short_name' => 4,
                'area' => 5,
                'series' => 6,
                'series_name' => 7,
                'unit_name' => 8,
                'year' => 9,
                'data_value' => 10,
                'min_year' => 11,
                'min_year_value' => 12,
                'max_year' => 13,
                'max_year_value' => 14,
                'average_max_year_value' => 15,
                'adjusted_growth_rate' => 16,
                'tagz' => 17,
                'tagz_number' => 18,
                'direction' => 19,
                'series_level' => 20,
                'goal_comparison' => 21,
                'star_level' => 22,
            ];
            
            // Open CSV file
            $handle = fopen($filePath, 'r');
            if ($handle === FALSE) {
                throw new \Exception("Could not open file: $filePath");
            }

            // Skip header row
            fgetcsv($handle);

            // Prepare batch insert
            $batchSize = 1000;
            $values = [];
            $placeholders = [];

            // Read CSV rows
            while (($row = fgetcsv($handle)) !== FALSE) {
                // Map the columns according to our specification
                $mappedData = [];
                foreach ($columnMapping as $field => $columnIndex) {
                    $mappedData[] = $row[$columnIndex] ?? null;
                }
                
                // Add sanitized values to the array
                $values = array_merge($values, array_map([$this, 'sanitizeValue'], $mappedData));
                $placeholders[] = '(' . implode(', ', array_fill(0, count($columnMapping), '?')) . ')';
                
                // Execute batch insert when batch size is reached
                if (count($placeholders) >= $batchSize) {
                    $this->executeBatchInsert($database, 'sdg_performers_data', array_keys($columnMapping), $values, $placeholders);
                    $values = [];
                    $placeholders = [];
                }
            }

            // Insert remaining rows
            if (!empty($placeholders)) {
                $this->executeBatchInsert($database, 'sdg_performers_data', array_keys($columnMapping), $values, $placeholders);
            }

            fclose($handle);

            return TRUE;
        } catch (\Exception $e) {
            if (isset($handle)) {
                fclose($handle);
            }
            \Drupal::logger('Import SDG Performers Excel')->error('Error importing Excel data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importSDGPerformersSummarySpreadsheet(string $filePath)
    {
        try {
            $database = Database::getConnection('default', 'custom');
            $database->query("TRUNCATE TABLE sdg_performers_summary_data");

            $columnMapping = [
                'ind_number' => 0,
                'ind_short_name' => 1,
                'goal_number' => 2,
                'target_number' => 3,
                'short_name' => 4,
                'area' => 5,
                'indicator_tagz_value' => 6,
                'indicator_tagz' => 7,
                'target_tagz' => 8,
                'goal_tagz_value' => 9,
                'goal_tagz' => 10,
                'goal_comparison' => 11,
                'star_level' => 12,
            ];

            // Open CSV file
            $handle = fopen($filePath, 'r');
            if ($handle === FALSE) {
                throw new \Exception("Could not open file: $filePath");
            }

            // Skip header row
            fgetcsv($handle);
            
            // Prepare batch insert
            $batchSize = 1000;
            $values = [];
            $placeholders = [];

            // Read CSV rows
            while (($row = fgetcsv($handle)) !== FALSE) {
                // Map the columns according to our specification
                $mappedData = [];
                foreach ($columnMapping as $field => $columnIndex) {
                    $mappedData[] = $row[$columnIndex] ?? null;
                }
                
                // Add sanitized values to the array
                $values = array_merge($values, array_map([$this, 'sanitizeValue'], $mappedData));
                $placeholders[] = '(' . implode(', ', array_fill(0, count($columnMapping), '?')) . ')';
                
                // Execute batch insert when batch size is reached
                if (count($placeholders) >= $batchSize) {
                    $this->executeBatchInsert($database, 'sdg_performers_summary_data', array_keys($columnMapping), $values, $placeholders);
                    $values = [];
                    $placeholders = [];
                }
            }

            // Insert remaining rows
            if (!empty($placeholders)) {
                $this->executeBatchInsert($database, 'sdg_performers_summary_data', array_keys($columnMapping), $values, $placeholders);
            }
            
            fclose($handle);

            return TRUE;
        } catch (\Exception $e) {
            if (isset($handle)) {
                fclose($handle);
            }

            \Drupal::logger('Import SDG Performers Summary Spreadsheet')->error('Error importing Spreadsheet data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importCountryGoalSnapshotDataExcel(string $filePath, string $countryCode, string $year)
    {
        try {
            $spreadsheet = IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $database = Database::getConnection('default', 'custom');

            $aggregatedData = [];


            foreach ($worksheet->getRowIterator() as $index => $row) {
                if ($index === 1) {
                    continue;
                }

                $rowData = [];
                foreach ($row->getCellIterator() as $cell) {
                    $rowData[] = $cell->getValue();
                }

                $goalNumber = $rowData[0];

                if (!isset($aggregatedData[$goalNumber])) {
                    $aggregatedData[$goalNumber] = [
                        'goal_number' => $goalNumber,
                        'grey_bar' => 0,
                        'snapshot_color' => '',
                        'indicator_percentage' => 0,
                        'snapshot_goal' => '',
                        'year' => $year,
                        'country_group' => $countryCode,
                    ];
                }


                $aggregatedData[$goalNumber]['grey_bar'] += (float)$rowData[1];
                if (empty($aggregatedData[$goalNumber]['snapshot_color']) && !empty($rowData[2])) {
                    $aggregatedData[$goalNumber]['snapshot_color'] = $rowData[2];
                }
                $aggregatedData[$goalNumber]['indicator_percentage'] += (float)$rowData[3];


                if (!empty($rowData[4])) {
                    $snapshotGoal = $this->normalizeSnapshotGoal($rowData[4]);
                    $aggregatedData[$goalNumber]['snapshot_goal'] = (float) $snapshotGoal;
                }
            }

            foreach ($aggregatedData as $data) {
                $existingRecord = $database->select('country_goal_snapshot_data', 'cg')
                ->fields('cg')
                ->condition('goal_number', $data['goal_number'])
                ->condition('year', $data['year'])
                ->condition('country_group', $data['country_group'])
                ->execute()
                    ->fetchAssoc();

                if ($existingRecord) {
                    $database->update('country_goal_snapshot_data_test')
                    ->fields([
                        'grey_bar' => $data['grey_bar'],
                        'snapshot_color' => $data['snapshot_color'],
                        'indicator_percentage' => $data['indicator_percentage'],
                        'snapshot_goal' => (float)$data['snapshot_goal'],
                    ])
                        ->condition('goal_number', $data['goal_number'])
                        ->condition('year', $data['year'])
                        ->condition('country_group', $data['country_group'])
                        ->execute();
                } else {
                    $database->insert('country_goal_snapshot_data_test')
                    ->fields([
                        'goal_number' => $data['goal_number'],
                        'grey_bar' => $data['grey_bar'],
                        'snapshot_color' => $data['snapshot_color'],
                        'indicator_percentage' => $data['indicator_percentage'],
                        'snapshot_goal' => (float)$data['snapshot_goal'],
                        'year' => $data['year'],
                        'country_group' => $data['country_group'],
                    ])
                        ->execute();
                }
            }
            return TRUE;
        } catch (\Exception $e) {
            \Drupal::logger('importCountryGoalSnapshotDataExcel')->error('Error importing Excel data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importCountryGoalSnapshotDataIrregularExcel(string $filePath, string $countryCode, string $year)
    {
        try {
            $spreadsheet = IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $database = Database::getConnection('default', 'custom');

            $aggregatedData = [];

            if ($countryCode === "PHL") {

                $transposedData = [];

                $highestRow = $worksheet->getHighestRow();
                $highestColumn = $worksheet->getHighestColumn();

                for ($row = 1; $row <= $highestRow; $row++) {
                    for ($col = 'A'; $col <= $highestColumn; $col++) {
                        $value = $worksheet->getCell($col . $row)->getValue();
                        $transposedData[$col][$row] = $value;
                    }
                }
                $formattedData = [];

                foreach ($transposedData as $col => $rowData) {
                    $formattedRow = [];
                    foreach ($rowData as $row => $value) {
                        $formattedRow[] = $value;
                    }

                    $formattedData[] = $formattedRow;
                }

                foreach ($formattedData as $row => $rowData) {
                    $goalNumber = "Goal " . $rowData[0] . " " . $rowData[1];
                    $snapshotGoal = isset($rowData[5]) ? (float)$rowData[5] : null;

                    $snapshotColor = 'Progress';
                    if ($snapshotGoal !== null && $snapshotGoal < 0) {
                        $snapshotColor = 'Regression';
                    }


                    $aggregatedData[$goalNumber] = [
                        'goal_number' => $goalNumber,
                        'grey_bar' => null,
                        'snapshot_color' => $snapshotColor,
                        'indicator_percentage' => null,
                        'snapshot_goal' => $snapshotGoal,
                        'year' => $year,
                        'country_group' => $countryCode,
                    ];
                }
                array_shift($aggregatedData);

            } else {

                foreach ($worksheet->getRowIterator() as $index => $row) {
                    if ($index === 1) {
                        continue;
                    }

                    $rowData = [];
                    foreach ($row->getCellIterator() as $cell) {
                        $rowData[] = $cell->getValue();
                    }

                    if ($countryCode === "MNG") {
                        $goalNumber = $rowData[0];
                        if (!isset($aggregatedData[$goalNumber])) {
                            $snapshotColor = 'Progress';
                            if ((float)$rowData[5] < 0) {
                                $snapshotColor = 'Regression';
                            }
                            $aggregatedData[$goalNumber] = [
                                'goal_number' => $goalNumber,
                                'grey_bar' => (float)$rowData[3],
                                'snapshot_color' => $snapshotColor,
                                'indicator_percentage' => 0,
                                'snapshot_goal' => (float)$rowData[5],
                                'year' => $year,
                                'country_group' => $countryCode,
                            ];
                        }
                    }
                }
            }


            foreach ($aggregatedData as $data) {

                $existingRecord = $database->select('country_goal_snapshot_data_test', 'cg')
                ->fields('cg')
                ->condition('goal_number', $data['goal_number'])
                ->condition('year', $data['year'])
                ->condition('country_group', $data['country_group'])
                ->execute()
                    ->fetchAssoc();

                if ($existingRecord) {
                    $database->update('country_goal_snapshot_data_test')
                    ->fields([
                        'grey_bar' => $data['grey_bar'],
                        'snapshot_color' => $data['snapshot_color'],
                        'indicator_percentage' => $data['indicator_percentage'],
                        'snapshot_goal' => (float)$data['snapshot_goal'],
                    ])
                        ->condition('goal_number', $data['goal_number'])
                        ->condition('year', $data['year'])
                        ->condition('country_group', $data['country_group'])
                        ->execute();
                } else {
                    $database->insert('country_goal_snapshot_data_test')
                    ->fields([
                        'goal_number' => $data['goal_number'],
                        'grey_bar' => $data['grey_bar'],
                        'snapshot_color' => $data['snapshot_color'],
                        'indicator_percentage' => $data['indicator_percentage'],
                        'snapshot_goal' => (float)$data['snapshot_goal'],
                        'year' => $data['year'],
                        'country_group' => $data['country_group'],
                    ])
                        ->execute();
                }
            }
            return TRUE;
        } catch (\Exception $e) {
            \Drupal::logger('importCountryGoalSnapshotDataIrregularExcel')->error('Error importing Excel data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importSDGDataAvailabilityExcel(string $filePath)
    {
        try {
            $spreadsheet = IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $database = Database::getConnection('default', 'custom');
            $yearColumnIndex = null;
            $headerValues = [];

            $headerRow = $worksheet->getRowIterator()->current();
            foreach ($headerRow->getCellIterator() as $cell) {
                $headerValues[] = $cell->getValue() ?? '';
            }

            $yearColumnIndex = array_search('Year', $headerValues);
            $columnLetter = Coordinate::stringFromColumnIndex($yearColumnIndex + 1);
            foreach ($worksheet->getRowIterator() as $index => $row) {
                if ($index === 1) {
                    continue;
                }

                $rowData = [];
                foreach ($headerValues as $header) {
                    $cellValue = $worksheet->getCell($columnLetter . 2)->getValue();
                    $rowData[$header] = $cellValue;
                }
                $year = $rowData['Year'];
            }

            foreach ($worksheet->getRowIterator() as $index => $row){
                $headerValues = [];
                if ($index === 1) {
                    continue;
                }
                if($year === 2019){
                    $rowData = [];
                    foreach ($row->getCellIterator() as $cell) {
                        $rowData[] = $cell->getValue();
                    }
                    $database->insert('data_availability_indicator')
                    ->fields([
                        'country_group' => $rowData[1],
                        'goal' => $rowData[2],
                        'target' => $rowData[3],
                        'indicator_number' => $rowData[4],
                        'indicator_order' => $rowData[5],
                        'indicator_short_name' => $rowData[6],
                        'tier' => $rowData[7],
                        'repeat' => $rowData[8],
                        'indicator_id' => $rowData[9],
                        'country_group_total' => $rowData[10],
                        'availability' => $rowData[11],
                        'year' => $rowData[12],
                        'availability_level' => $rowData[13]
                    ])
                        ->execute();
                }

                elseif($year === 2020){
                    $rowData = [];
                    foreach ($row->getCellIterator() as $cell) {
                        $rowData[] = $cell->getValue();
                    }

                    $database->insert('data_availability_indicator')
                    ->fields(
                        [
                            'country_group' => $rowData[0],
                            'goal' => $rowData[1],
                            'target' => $rowData[2],
                            'indicator_number' => $rowData[3],
                            'indicator_order' => $rowData[4],
                            'indicator_short_name' => $rowData[5],
                            'tier' => $rowData[6],
                            'repeat' => $rowData[7],
                            'indicator_id' => $rowData[8],
                            'country_group_total' => $rowData[9],
                            'availability' => $rowData[10],
                            'year' => $rowData[11],
                            'availability_level' => $rowData[12]
                        ]
                    )
                        ->execute();
                }

                elseif ($year === 2021) {
                    $rowData = [];
                    foreach ($row->getCellIterator() as $cell) {
                        $rowData[] = $cell->getValue();
                    }

                    $database->insert('data_availability_indicator')
                    ->fields(
                        [
                            'country_group' => $rowData[0],
                            'goal' => $rowData[1],
                            'target' => $rowData[2],
                            'indicator_number' => $rowData[3],
                            'indicator_order' => $rowData[4],
                            'indicator_short_name' => $rowData[5],
                            'tier' => $rowData[6],
                            'repeat' => $rowData[7],
                            'source_category_primary' => $rowData[8],
                            'source_category_secondary' => $rowData[9],
                            'custodian' => $rowData[10],
                            'custodian_partner' => $rowData[11],
                            'country_group_label' => $rowData[12],
                            'country_group_total' => $rowData[13],
                            'availability' => $rowData[14],
                            'average_delay' => $rowData[15],
                            'year' => $rowData[16],
                            'availability_level' => $rowData[17]
                        ]
                    )
                        ->execute();
                }

                elseif ($year === 2022) {
                    $rowData = [];
                    foreach ($row->getCellIterator() as $cell) {
                        $rowData[] = $cell->getValue();
                    }

                    $database->insert('data_availability_indicator')
                    ->fields(
                        [
                            'goal' => $rowData[0],
                            'target' => $rowData[1],
                            'indicator_number' => $rowData[2],
                            'indicator_order' => $rowData[3],
                            'indicator_short_name' => $rowData[4],
                            'repeat' => $rowData[5],
                            'country_group' => $rowData[6],
                            'country_group_label' => $rowData[7],
                            'country_group_total' => $rowData[8],
                            'availability' => $rowData[9],
                            'average_delay' => $rowData[10],
                            'latest_year' => $rowData[11],
                            'year' => $rowData[12],
                            'availability_level' => $rowData[13]
                        ]
                    )
                        ->execute();
                }

                elseif ($year === 2023) {
                    $rowData = [];
                    foreach ($row->getCellIterator() as $cell) {
                        $rowData[] = $cell->getValue();
                    }

                    $database->insert('data_availability_indicator')
                    ->fields([
                        'goal' => $rowData[0],
                        'target' => $rowData[1],
                        'indicator_number' => $rowData[2],
                        'indicator_order' => $rowData[3],
                        'indicator_short_name' => $rowData[4],
                        'repeat' => $rowData[5],
                        'country_group' => $rowData[6],
                        'country_group_label' => $rowData[7],
                        'country_group_total' => $rowData[8],
                        'top_series' => $rowData[9],
                        'top_series_2015' => $rowData[10],
                        'availability' => $rowData[11],
                        'availability_two_dp_2015' => $rowData[12],
                        'availability_one_dp' => $rowData[13],
                        'availability_one_dp_2015' => $rowData[14],
                        'average_delay' => $rowData[15],
                        'latest_year' => $rowData[16],
                        'year' => $rowData[17],
                        'availability_level' => $rowData[18],
                        'availability_level_2015' => $rowData[19]
                    ])
                        ->execute();
                }
                else {
                    $columnMappings = [
                        'Goal' => 'goal',
                        'Target' => 'target',
                        'IND_Number' => 'indicator_number',
                        'IND_short_name' => 'indicator_short_name',
                        'CountryGroup' => 'country_group',
                        'Year' => 'year',
                        'Repeat' => 'repeat',
                        'CountryGroupTotal' => 'country_group_total',
                        'AvailabilityLevel' => 'availability_level',
                        'Availability' => 'availability'
                    ];
                
                    $rowData = [];
                    foreach ($columnMappings as $excelHeader => $dbColumn) {
                        
                        $columnIndex = array_search($excelHeader, $headerValues);
                        if ($columnIndex === false) {
                            continue;
                        }
                        // convert column index to letter
                        $columnLetter = Coordinate::stringFromColumnIndex($columnIndex + 1);
                        
                        $rowIndex = $row->getRowIndex();
                        $cellAddress = $columnLetter . $rowIndex;
                        
                        $cellValue = $worksheet->getCell($cellAddress)->getValue();
                        $rowData[$dbColumn] = $cellValue;
                    }
                
                    $database->insert('data_availability_indicator')
                        ->fields([
                            'country_group' => $rowData['country_group'] ?? null,
                            'indicator_number' => $rowData['indicator_number'] ?? null,
                            'indicator_short_name' => $rowData['indicator_short_name'] ?? null,
                            'goal' => $rowData['goal'] ?? null,
                            'target' => $rowData['target'] ?? null,
                            'year' => $rowData['year'] ?? null,
                            'repeat' => $rowData['repeat'] ?? null,
                            'country_group_total' => $rowData['country_group_total'] ?? null,
                            'availability_level' => $rowData['availability_level'] ?? null,
                            'availability' => $rowData['availability'] ?? null,
                        ])
                        ->execute();
                }
            }
            return TRUE;
        } catch (\Exception $e) {
            \Drupal::logger('your_module')->error('Error importing Excel data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importSDGDataSourcesExcel(string $filePath)
    {
        try {
            $spreadsheet = IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $database = Database::getConnection('default', 'custom');
            foreach ($worksheet->getRowIterator() as $index => $row) {
                if ($index === 1) {
                    continue;
                }

                $rowData = [];
                foreach ($row->getCellIterator() as $cell) {
                    $rowData[] = $cell->getValue();
                }

                $database->insert('data_source')
                ->fields([
                    'indicator_number' => $rowData[0],
                    'indicator_code' => $rowData[1],
                    'indicator_name' => $rowData[2],
                    'compiler' => $rowData[3],
                    'compiler_type' => $rowData[4],
                    'source' => $rowData[5],
                    'source_module' => $rowData[6],
                    'source_category' => $rowData[7],
                    'source_footnote' => $rowData[8],
                    'source_resources' => $rowData[9]
                ])
                    ->execute();
            }
            return TRUE;
        } catch (\Exception $e) {
            \Drupal::logger('your_module')->error('Error importing Excel data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importCountryGoalSnapshotTestDataExcel(string $filePath, string $countryCode, string $year)
    {
        try {
            $spreadsheet = IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $database = Database::getConnection('default', 'custom');

            // Delete existing records for the given country and year
            $database->delete('country_goal_snapshot_data_test')
                ->condition('year', $year)
                ->condition('country_group', $countryCode)
                ->execute();

            $aggregatedData = [];

            if ($countryCode === "PHL") {
                $transposedData = [];
                $highestRow = $worksheet->getHighestRow();
                $highestColumn = $worksheet->getHighestColumn();

                for ($row = 1; $row <= $highestRow; $row++) {
                    for ($col = 'A'; $col <= $highestColumn; $col++) {
                        $value = $worksheet->getCell($col . $row)->getValue();
                        $transposedData[$col][$row] = $value;
                    }
                }

                $formattedData = [];
                foreach ($transposedData as $col => $rowData) {
                    $formattedRow = [];
                    foreach ($rowData as $row => $value) {
                        $formattedRow[] = $value;
                    }
                    $formattedData[] = $formattedRow;
                }

                foreach ($formattedData as $row => $rowData) {
                    $goalNumber = "Goal " . $rowData[0] . " " . $rowData[1];
                    $snapshotGoal = isset($rowData[5]) ? (float)$rowData[5] : null;
                    $snapshotColor = 'Progress';
                    if ($snapshotGoal !== null && $snapshotGoal < 0) {
                        $snapshotColor = 'Regression';
                    }

                    $aggregatedData[$goalNumber] = [
                        'goal_number' => $goalNumber,
                        'grey_bar' => null,
                        'snapshot_color' => $snapshotColor,
                        'indicator_percentage' => null,
                        'snapshot_goal' => $snapshotGoal,
                        'year' => $year,
                        'country_group' => $countryCode,
                    ];
                }
                array_shift($aggregatedData);
            } else {
                foreach ($worksheet->getRowIterator() as $index => $row) {
                    if ($index === 1) {
                        continue;
                    }

                    $rowData = [];
                    foreach ($row->getCellIterator() as $cell) {
                        $rowData[] = $cell->getValue();
                    }

                    if ($countryCode === "MNG") {
                        $goalNumber = $rowData[0];
                        $snapshotColor = 'Progress';
                        if ((float)$rowData[5] < 0) {
                            $snapshotColor = 'Regression';
                        }
                        $aggregatedData[$goalNumber] = [
                            'goal_number' => $goalNumber,
                            'grey_bar' => (float)$rowData[3],
                            'snapshot_color' => $snapshotColor,
                            'indicator_percentage' => 0,
                            'snapshot_goal' => (float)$rowData[5],
                            'year' => $year,
                            'country_group' => $countryCode,
                        ];
                    } else {
                        $goalNumber = $rowData[0];
                        if (!isset($aggregatedData[$goalNumber])) {
                            $aggregatedData[$goalNumber] = [
                                'goal_number' => $goalNumber,
                                'grey_bar' => 0,
                                'snapshot_color' => '',
                                'indicator_percentage' => 0,
                                'snapshot_goal' => '',
                                'year' => $year,
                                'country_group' => $countryCode,
                            ];
                        }

                        $aggregatedData[$goalNumber]['grey_bar'] += (float)$rowData[1];
                        if (empty($aggregatedData[$goalNumber]['snapshot_color']) && !empty($rowData[2])) {
                            $aggregatedData[$goalNumber]['snapshot_color'] = $rowData[2];
                        }
                        $aggregatedData[$goalNumber]['indicator_percentage'] += (float)$rowData[3];

                        if (!empty($rowData[4])) {
                            $snapshotGoal = $this->normalizeSnapshotGoal($rowData[4]);
                            $aggregatedData[$goalNumber]['snapshot_goal'] = (float) $snapshotGoal;
                        }
                    }
                }
            }

            foreach ($aggregatedData as $data) {
                $database->insert('country_goal_snapshot_data_test')
                    ->fields([
                        'goal_number' => $data['goal_number'],
                        'grey_bar' => $data['grey_bar'],
                        'snapshot_color' => $data['snapshot_color'],
                        'indicator_percentage' => $data['indicator_percentage'],
                        'snapshot_goal' => (float)$data['snapshot_goal'],
                        'year' => $data['year'],
                        'country_group' => $data['country_group'],
                    ])
                    ->execute();
            }

            return TRUE;
        } catch (\Exception $e) {
            \Drupal::logger('your_module')->error('Error importing Excel data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importGoalSnapshot(string $filePath, string $year, string $country_group)
    {
        try {
            $database = Database::getConnection('default', 'custom');
            
            // Delete existing records for the given country and year
            $database->delete('country_goal_snapshot_data')
                ->condition('year', $year)
                ->condition('country_group', $country_group)
                ->execute();

            // Detect the delimiter
            $delimiter = $this->detectDelimiter($filePath);

            // Open CSV file with tab delimiter
            $handle = fopen($filePath, 'r');
            if ($handle === FALSE) {
                throw new \Exception("Could not open file: $filePath");
            }
            
            // Skip header row
            fgetcsv($handle, 0, $delimiter);
            
            // Read rows
            while (($row = fgetcsv($handle, 0, $delimiter)) !== FALSE) {
                
                // handling specific situations where snapshot color and snapshot goal might be null, if its null 
                //  this will prevent the sdg-tracker-charts form visualizing correctly
                $snapshot_color = $this->sanitizeValueGoalSnapshot($row[2]);
                $snapshot_goal = $this->sanitizeValueGoalSnapshot($row[4]);
                if ($snapshot_color === null) {
                    $snapshot_color = "Insufficient Indicators";
                }
                if ($snapshot_goal === null) {
                    $snapshot_goal = 0;
                }

                $database->insert('country_goal_snapshot_data')
                    ->fields([
                        'goal_number' => $this->sanitizeValueGoalSnapshot($row[0]),
                        'grey_bar' => $this->sanitizeValueGoalSnapshot($row[1], 'grey_bar'),
                        'snapshot_color' => $snapshot_color,
                        'indicator_percentage' => $this->sanitizeValueGoalSnapshot($row[3], 'indicator_percentage'),
                        'snapshot_goal' => $snapshot_goal,
                        'year' => $year,
                        'country_group' => $country_group,
                    ])
                    ->execute();
            }

            fclose($handle);

            // Delete records where goal_number is empty string OR NULL
            $query = "DELETE FROM {country_goal_snapshot_data} 
                WHERE country_group = ? 
                AND (goal_number = '' OR goal_number IS NULL)";

            $database->query($query, [$country_group]);

            return TRUE;
        } catch (\Exception $e) {
            if (isset($handle)) {
                fclose($handle);
            }
            \Drupal::logger('Import Goal Snapshot Data (import National-SDG-Tracker) Spreadsheet')
                ->error('Error importing CSV data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }

    public function importDataAvailabilitySpreadsheet(string $filePath)
    {
        try {
            $database = Database::getConnection('default', 'custom');
            $database->query("TRUNCATE TABLE data_availability_indicator");
            
            $columnMapping = [
                'goal' => 0,
                'target' => 1,
                'indicator_number' => 2,
                'indicator_order' => 3,
                'indicator_short_name' => 4,
                '`repeat`' => 5,
                'country_group' => 6,
                'country_group_label' => 7,
                'country_group_total' => 8,
                'availability' => 9,
                'availability_two_dp_2015' => 10,
                'availability_one_dp' => 11,
                'availability_one_dp_2015' => 12,
                'average_delay' => 13,
                'latest_year' => 14,
                'group_unapplicable' => 15,
                'year' => 16,
                'availability_level' => 17,
                'availability_level_2015' => 18
            ];
            
            // Open CSV file
            $handle = fopen($filePath, 'r');
            if ($handle === FALSE) {
                throw new \Exception("Could not open file: $filePath");
            }
            
            // Skip header row
            fgetcsv($handle);
            
            // Prepare batch insert
            $batchSize = 1000;
            $values = [];
            $placeholders = [];
            
            // Read CSV rows
            while (($row = fgetcsv($handle)) !== FALSE) {
                // Map the columns according to our specification
                $mappedData = [];
                foreach ($columnMapping as $field => $columnIndex) {
                    $value = $row[$columnIndex] ?? null;
                    // Special handling for group_unapplicable column
                    if ($field === 'group_unapplicable' && ($value === 'NA' || $value === null)) {
                        $value = 0;
                    }
                    $mappedData[] = $value;
                }
                
                // Add sanitized values to the array
                $values = array_merge($values, array_map([$this, 'sanitizeValue'], $mappedData));
                $placeholders[] = '(' . implode(', ', array_fill(0, count($columnMapping), '?')) . ')';
                
                // Execute batch insert when batch size is reached
                if (count($placeholders) >= $batchSize) {
                    $this->executeBatchInsert($database, 'data_availability_indicator', array_keys($columnMapping), $values, $placeholders);
                    $values = [];
                    $placeholders = [];
                }
            }
            
            // Insert remaining rows
            if (!empty($placeholders)) {
                $this->executeBatchInsert($database, 'data_availability_indicator', array_keys($columnMapping), $values, $placeholders);
            }
            
            fclose($handle);
            
            return TRUE;
        } catch (\Exception $e) {
            if (isset($handle)) {
                fclose($handle);
            }
            \Drupal::logger('Import Data Availability Test')
                ->error('Error importing CSV data: @error', ['@error' => $e->getMessage()]);
            return FALSE;
        }
    }
    
}