<?php

namespace Drupal\unsdg_module\Services\SdgDataPointsServices;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryFactoryInterface;
use Drupal\unsdg_module\Services\ContentServices\GoalService;
use Drupal\unsdg_module\Services\ContentServices\RegionService;
use Drupal\Core\Database\Database;
use Drupal\unsdg_module\Services\SdgDataPointsServices\Models\CountryGoalSnapshotData;

class DataPointsService
{
    protected $queryFactory;
    protected $entityTypeManager;
    protected $goalService;
    protected $regionService;

    public function __construct(
        EntityTypeManagerInterface $entityTypeManager,
        QueryFactoryInterface $queryFactory,
        GoalService $goalService,
        RegionService $regionService
    ) {
        $this->entityTypeManager = $entityTypeManager;
        $this->queryFactory = $queryFactory;
        $this->goalService = $goalService;
        $this->regionService = $regionService;
    }

    public function fetchRegionalAnalysisDashboardData($region)
    {
        $goals = $this->goalService->getAllGoals();
        $targetData = $this->getRegionTargets($goals, $region);
        
        $dashboardData = [];
        foreach ($targetData as $target) {
            $category = $target['goal_number'];
            $goal = $this->goalService->getGoalByGoalNumber($category);
            $dashboardData[] = [
                "goal_number" => $target['goal_number'],
                "goal_name" => $goal ? $goal->label() : '',
                "image_url"=> isset($goal->image_url) ? $goal->image_url : null,
                "target_number"=> $target["target_number"],
                "target_color"=> $target["target_color"],
                "target_long_name"=> $target["target_long_name"],
            ];
        }

        return $dashboardData;
    }

    public function getRegionTargets($goals, $region)
    {
        $database = Database::getConnection('default', 'custom');

        $subquery = $database->select('goal_data', 'c')
            ->fields('c', ['year'])
            ->condition('c.country_group', $region)
            ->orderBy('c.year', 'DESC')
            ->range(0, 1);
        $highestYear = $subquery->execute()->fetchField();

        $query = $database->select('target', 'c')
            ->fields('c')
            ->condition('c.country_group', $region)
            ->condition('c.year', $highestYear)
            ->orderBy('c.target_color')
            ->orderBy('c.target_num', 'ASC');

        $results = $query->execute()->fetchAll();
        //return $results;
        $regionalAnalysisDashboardData = [];
        foreach ($goals as $goal) {
            $goalNumber = $goal->field_goal_number->value;
            foreach ($results as $data) {
                if ($goalNumber == $data->goal_num) {
                    $regionalAnalysisDashboardData[] = [
                        'goal_number' => $goal->field_goal_number->value,
                        'target_number' => $data->target_num,
                        'target_color' => $data->target_color,
                        'target_long_name' => $data->target_long_name,
                    ];
                }
            }
        }

        return $regionalAnalysisDashboardData;
    }

    public function getRegionTargetsByGoalId($goalNumber, $region)
    {
        $database = Database::getConnection('default', 'custom');

        // $subquery = $database->select('goal_data', 'sub')
        //     ->fields('sub', ['year'])
        //     ->condition('sub.goal_number', $goalNumber)
        //     ->condition('c.country_group', $region)
        //     ->condition('sub.year', 2030, '<>')
        //     ->orderBy('sub.year', 'DESC')
        //     ->range(0, 1);

        $query = $database->select('target', 'c')
            ->fields('c')
            ->condition('c.goal_num', $goalNumber)
            ->condition('c.country_group', $region)

            ->condition('c.year', 2030, '=')
            ->orderBy('c.target_color')
            ->orderBy('c.target_num', 'ASC');

        $results = $query->execute()->fetchAllAssoc('target_num');

        return $results;
    }

    // public function getIndicatorDataByGoalId($goalNumber)
    // {
    //     $database = Database::getConnection('default', 'custom');

    //     $query = $database->select('target', 'c')
    //         ->fields('c') // Select specific fields
    //         ->condition('c.goal_num', $goalNumber)
    //         ->orderBy('c.target_color')
    //         ->orderBy('c.target_num', 'ASC');

    //     // Execute the query and fetch all results
    //     $results = $query->execute()->fetchAll();

    //     return $results;
    // }

    public function getTargetIndicatorDataByGoalId($goalNumber, $region, $year)
    {
        $database = Database::getConnection('default', 'custom');
        $targets = $this->getRegionTargetsByGoalId($goalNumber, $region);
        $targetIndicatorData = [];

        $query = $database->select('indicator_data', 'c')
            ->fields('c')
            ->condition('c.snapshot_indicator', 'NA', '!=')
            ->condition('c.snapshot_indicator', '0', '!=');
        //->condition('c.indicator_id', $indicator->indicator_number);

        if ($region) {
            $query->condition('c.country_group', $region);
        }

        if ($year == null) {
            $subquery = $database->select('indicator_data', 'sub')
                ->orderBy('sub.year', 'DESC')
                ->range(0, 1);
            $subquery->addExpression('MAX(sub.year)');

            $query->condition('c.year', $subquery, '=');
        } elseif ($year) {
            $query->condition('c.year', $year);
        }

        $indicatorData = $query->execute()->fetchAll();

        foreach ($targets as $target) {
            $target_number = $target->target_num;
            $indicators = $this->getIndicatorsByTargetId($target_number, $region);
            $targetData = [
                'target' => $target,
                'indicators' => [],
            ];

            foreach ($indicators as $indicator) {
                foreach ($indicatorData as $dataPoint) {
                    if ($dataPoint->indicator_id == $indicator->indicator_number) {
                        $targetData['indicators'][] = [
                            'indicator' => $indicator,
                            'data' => $dataPoint,
                        ];
                    }
                }
            }
            $targetIndicatorData[] = $targetData;
        }

        return $targetIndicatorData;
    }

    public function getTargetIndicatorDashboardDetailData($goalNumber, $region, $year)
    {
        $database = Database::getConnection('default', 'custom');
        $targetIndicatorData = [];
        $query = $database->select('indicator_data', 'c')
            ->fields('c')
            ->condition('c.snapshot_indicator', 'NA', '!=');
        if ($region) {
            $query->condition('c.country_group', $region);
        }
        if ($year) {
            $query->condition('c.year', $year);
        }
        $indicatorData = $query->execute()->fetchAll();
        $targets = $this->getRegionTargetsByGoalId($goalNumber, $region);
        foreach ($targets as $target) {
            $target_number = $target->target_num;
            $indicators = $this->getIndicatorsByTargetId($target_number, $region);
            $results = [];
            foreach ($indicators as $indicator) {
                foreach ($indicatorData as $dataPoint) {
                    if ($indicator->indicator_number === $dataPoint->indicator_id) {
                        $dataPoint->indicator_short_name = $indicator->indicator_short_name;
                        $results[] = $dataPoint;
                    }
                }
            }

            foreach ($results as $record) {
                $targetIndicatorData[] = [
                    'target_color' => $target->target_color,
                    'indicator_data_id' => $record->indicator_data_id,
                    'indicator_id' => $record->indicator_id,
                    'indicator_short_name' => $record->indicator_short_name,
                    'year' => $record->year,
                    'snapshot_indicator' => $record->snapshot_indicator,
                    'country_group' => $record->country_group
                ];
            }
        }

        $colorOrder = [
            '1-Green' => 1,
            '2-Yellow' => 2,
            '3-Red' => 3,
            '4-Grey' => 4
        ];

        usort($targetIndicatorData, function ($a, $b) use ($colorOrder) {
            $snapshotComparison = $b['snapshot_indicator'] <=> $a['snapshot_indicator'];

            return $snapshotComparison !== 0
                ? $snapshotComparison
                : $colorOrder[$a['target_color']] <=> $colorOrder[$b['target_color']];
        });
            
        // usort($targetIndicatorData, function ($a, $b) use ($colorOrder) {
        //     return $colorOrder[$a['target_color']] <=> $colorOrder[$b['target_color']];
        // });

        return $targetIndicatorData;
    }

    public function getIndicatorsByTargetId($targetNumber)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('indicator', 'c')
            ->fields('c')
            ->condition('c.target_number', $targetNumber);

        $results = $query->execute()->fetchAll();
        return $results;
    }

    public function fetchLatestRegionalAnalysisSnapshotChartData($region, $goalNumber)
    {
        $regionalAnalysisSnapshotChartData = [];

        $goal = $this->goalService->getGoalByGoalNumber($goalNumber);
        //return $goal->image_url;
        $targetData = $this->getTargetIndicatorDataByGoalId($goalNumber, $region, null);
        //return $targetData;
        foreach ($targetData as $data) {
            foreach ($data['indicators'] as $indicatorDetail)
                $regionalAnalysisSnapshotChartData[] = [
                    'goal_name' => $goal ? $goal->label() : '',
                    'image_url' => isset($goal->image_url) ? $goal->image_url : null,
                    'goal_number' => $goal->field_goal_number->value,
                    'target_number' => $indicatorDetail['indicator']->indicator_number,
                    'snapshot_indicator' => $indicatorDetail['data']->snapshot_indicator,
                    'target_long_name' => $indicatorDetail['indicator']->indicator_short_name
                ];
        }

        usort($regionalAnalysisSnapshotChartData, function ($a, $b) {        
            $snapshotComparison = $b['snapshot_indicator'] <=> $a['snapshot_indicator'];
            return $snapshotComparison !== 0
                ? $snapshotComparison
                : $this->compareTargetNumbers($a['target_number'], $b['target_number']);
        });

        // usort($regionalAnalysisSnapshotChartData, function ($a, $b) {
        //     return $this->compareTargetNumbers($a['target_number'], $b['target_number']);
        // });
        return $regionalAnalysisSnapshotChartData;
    }

    public function getGoalDataByGoalId($goalNumber, $countryGroup)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('goal_data', 'c')
            ->fields('c')
            ->condition('c.country_group', $countryGroup);

        if ($goalNumber) {
            $query = $query->condition('c.goal_number', $goalNumber);
        }

        $results = $query->execute()->fetchAll();
        return $results;
    }

    public function getAllGoalData($countryGroup)
    {
        $goals = $this->goalService->getAllGoals();
        $regionalAnalysisSnapshotChartData = [];
        $currentYear = date('Y'); //get current year (to address data set that will come in for year 2030)
        $database = Database::getConnection('default', 'custom');
        $subquery = $database->select('goal_data', 'c')
            ->fields('c', ['year'])
            ->condition('c.year', 2030, '<>')
            ->orderBy('c.year', 'DESC')
            ->range(0, 1);
        $highestYear = $subquery->execute()->fetchField();

        $query = $database->select('goal_data', 'c')
            ->fields('c')
            ->condition('c.country_group', $countryGroup)
            //->condition('c.goal_number', $goalItem->field_goal_number->value)
            ->condition('c.year', $highestYear);

        $results = $query->execute()->fetchAll();
        //return $results;
        foreach ($goals as $goalNode) {
            $goal = $this->goalService->getGoalDetailsByGoalNode($goalNode);
            foreach ($results as $result) {
                if ($result->goal_number == $goal->goal_number) {
                    $filteredResult = (object) array_filter((array) $result, function ($key) {
                        //exclude properties as they are not used
                        $excludeProperties = ['dashboard_goal', 'snapshot_count', 'indicator_count',];
                        return !in_array($key, $excludeProperties);
                    }, ARRAY_FILTER_USE_KEY);

                    $filteredResult->goal_name = $goal ? $goal->goal_name : '';
                    $filteredResult->image_url = isset($goal->image_url) ? $goal->image_url : null;
                    $regionalAnalysisSnapshotChartData[] = $filteredResult;
                }
            }
        }

        return $regionalAnalysisSnapshotChartData;
    }

    public function fetchRegionalDataAvailabilityByRegion($countryGroup, $year)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('data_availability_indicator', 'c')
            ->fields('c')
            ->condition('c.country_group', $countryGroup)
            ->condition('c.repeat', 'NA');

        if ($year) {
            $query = $query->condition('c.year', $year);
        }

        $results = $query->execute()->fetchAll();
        return $results;
    }

    public function calculateRegionalDataAvailabilityByRegionCountryGroups()
    {
        $countryGroupList = $this->regionService->getCountries();
        if ($countryGroupList === null || count($countryGroupList) === 0) {
            return null;
        }

        $regionalDataAvailability = [];

        foreach ($countryGroupList as $country) {
            $countryGroupIso = $country['isoCode'];
            $year = $this->fetchLatestDataAvailabilityYear($countryGroupIso);
            $database = Database::getConnection('default', 'custom');

            $query = $database->select('data_availability_indicator', 'c')
                ->fields('c')
                ->condition('c.country_group', $countryGroupIso)
                ->condition('c.year', $year);

            $results = $query->execute()->fetchAll();
            $countryData = $this->groupRegionalDataAvailabilityByCountryGroup($results);
            $totalCount = array_sum($countryData);
            $result = [
                'country' => $country['name'],
                'count' => $totalCount,
                'isoCode' => $countryGroupIso
            ];

            foreach ($countryData as $category => $count) {
                switch ($category) {
                    case 'Sufficient':
                        $result['Sufficient'] = $count;
                        break;
                    case 'No Data':
                        $result['No Data'] = $count;
                        break;
                    case 'Insufficient':
                        $result['Insufficient Data'] = $count;
                        break;
                    case 'Not Applicable':
                        $result['Not Applicable'] = $count;
                        break;
                }
            }

            $regionalDataAvailability[] = $result;
        }
        $this->updateCountryDataAvailabilityStatistics($regionalDataAvailability, $year);
        return $regionalDataAvailability;
    }

    public function updateCountryDataAvailabilityStatistics($regionalDataAvailability, $year)
    {
        $database = Database::getConnection('default', 'custom');
        foreach ($regionalDataAvailability as $countryDataAvailability) {
            $data = [
                'country' => $countryDataAvailability['country'],
                'iso_code' => $countryDataAvailability['isoCode'],
                'count' => $countryDataAvailability['count'] ?? 0,
                'sufficient' => $countryDataAvailability['Sufficient'] ?? 0,
                'no_data' => $countryDataAvailability['No Data'] ?? 0,
                'insufficient_data' => $countryDataAvailability['Insufficient Data'] ?? 0,
                //'not_applicable' => $countryDataAvailability['Not Applicable'] ?? 0,
                'year' => $year ?? 0,
            ];


            $database->merge('country_data_availability_data')
                ->keys([
                    'iso_code' => $countryDataAvailability['isoCode'],
                    'year' => $year ?? 0
                ])
                ->fields($data)
                ->execute();
        }

        return $regionalDataAvailability;
    }

    public function fetchLatestRegionalDataAvailabilityByRegionCountry($countryGroup)
    {
        $countryGroupList = $this->regionService->getCountriesFromCountryGroupByCountryName($countryGroup);
        if ($countryGroup === null) {
            return null;
        }
        $database = Database::getConnection('default', 'custom');
        $regionalDataAvailability = [];
        $query = $database->select('data_availability_indicator', 'c')
            ->fields('c', ['year'])
            ->orderBy('c.year', 'DESC')
            ->range(0, 1); // limit to the first result

        $highestYear = $query->execute()->fetchField();
        $excludeProperties = [
            'country_data_availability_data_id',
            'year'
        ];

        $regionalDataAvailability = [];
        $query = $database->select('country_data_availability_data', 'c')
            ->fields('c')
            ->condition('c.year', $highestYear);
        $dataPoints = $query->execute()->fetchAll();

        foreach ($countryGroupList as $country) {
            $countryGroupIso = $country['isoCode'];
            foreach ($dataPoints as $element) {
                if ($countryGroupIso == $element->iso_code) {
                    $regionalDataAvailability[] = [
                        'country' => $element->country,
                        'count' => $element->count,
                        'Sufficient' => $element->sufficient,
                        'Insufficient Data' => $element->insufficient_data,
                        'No Data' => $element->no_data,
                    ];
                }
            }
        }
        return $regionalDataAvailability;
    }

    public function fetchLatestDataAvailabilityYear($countryGroup)
    {
        $database = Database::getConnection('default', 'custom');
        //get latest indicator available year
        $query = $database->select('data_availability_indicator', 'c')
            ->fields('c', ['year'])
            ->condition('c.country_group', $countryGroup)
            ->orderBy('c.year', 'DESC')
            ->range(0, 1); // limit to the first result

        $highestYear = $query->execute()->fetchField();
        return $highestYear;
    }

    public function groupRegionalDataAvailabilityByYear($regionalDataAvailability)
    {
        $groupedResults = [];
        foreach ($regionalDataAvailability as $result) {
            $year = $result->year;
            $availabilityLevel = ($year > 2022) ? $result->availability_level_2015 : $result->availability_level;

            if (!isset($groupedResults[$year])) {
                $groupedResults[$year] = [];
            }
            if (!isset($groupedResults[$year][$availabilityLevel])) {
                $groupedResults[$year][$availabilityLevel] = 0;
            }
            $groupedResults[$year][$availabilityLevel]++;
        }

        $transformedResults = [];
        foreach ($groupedResults as $year => $availabilityLevels) {
            $transformedResults[] = [
                'year' => $year,
                'availability' => $availabilityLevels
            ];
        }
        usort($transformedResults, function ($a, $b) {
            return $a['year'] <=> $b['year'];
        });
        return $transformedResults;
    }

    public function groupRegionalDataAvailabilityByCountryGroup($regionalDataAvailability)
    {
        $groupedResults = [];

        foreach ($regionalDataAvailability as $result) {
            $year = $result->year;
            $availabilityLevel = ($year > 2022) ? $result->availability_level_2015 : $result->availability_level;
            if (!isset($groupedResults[$availabilityLevel])) {
                $groupedResults[$availabilityLevel] = 0;
            }
            $groupedResults[$availabilityLevel]++;
        }
        return $groupedResults;
    }

    public function groupRegionalDataAvailabilityByGoal($regionalDataAvailability)
    {
        $groupedResults = [];
        $goalCache = [];

        foreach ($regionalDataAvailability as $dataPoint) {
            $goalNumber = $dataPoint->goal;
            sscanf($goalNumber, "Goal %d", $goalNumber);
            $goalNumber = intval($goalNumber);
            if (!isset($goalCache[$goalNumber])) {
                $goal = $this->goalService->getGoalByGoalNumber($goalNumber);
                $goalCache[$goalNumber] = $goal;
            } else {
                $goal = $goalCache[$goalNumber];
            }

            $year = $dataPoint->year;
            $availabilityLevel = ($year > 2022) ? $dataPoint->availability_level_2015 : $dataPoint->availability_level;
            if (!isset($groupedResults[$goalNumber])) {
                $groupedResults[$goalNumber] = [
                    'goal_number' => $goalNumber,
                    'goal_name' => $goal ? $goal->label() : '',
                    'image_url' => isset($goal->image_url) ? $goal->image_url : null,
                    'dataAvailability' => [],
                ];
            }

            if (!isset($groupedResults[$goalNumber]['dataAvailability'][$availabilityLevel])) {
                $groupedResults[$goalNumber]['dataAvailability'][$availabilityLevel] = [
                    'count' => 0,
                ];
            }
            $groupedResults[$goalNumber]['dataAvailability'][$availabilityLevel]['count']++;
        }

        foreach ($groupedResults as &$goal) {
            $total = array_sum(array_column($goal['dataAvailability'], 'count'));

            foreach ($goal['dataAvailability'] as &$data) {
                $data['percentage'] = ($total > 0) ? round(($data['count'] / $total) * 100, 2) : 0;
            }
        }
        uasort($groupedResults, function ($a, $b) {
            return $a['goal_number'] - $b['goal_number'];
        });
        return array_values($groupedResults);
    }

    public function fetchRegionalIndicatorDetailsDataAvailabilityByRegionGoal($countryGroup, $goal, $year)
    {
        $countryIsoList = $this->regionService->getCountriesIsoFromCountryGroupName($countryGroup);
        if ($countryIsoList === null) {
            return null;
        }
        $database = Database::getConnection('default', 'custom');
        $results = [];


        $query = $database->select('data_availability_indicator', 'c')
            ->fields('c')
            ->condition('c.goal', $goal);
        if ($year) {
            $query->condition('c.year', $year);
        }
        $result = $query->execute()->fetchAll();
        $dataAvailabilityIndicator = array_merge($results, $result);

        foreach ($countryIsoList as $countryIso) {
            foreach ($dataAvailabilityIndicator as $element) {
                if ($countryIso == $element->country_group) {
                    $results[] = $element;
                }
            }
        }
        return $results;
    }

    public function groupRegionalIndicatorDetailsDataAvailabilityByIndicator($regionalDataAvailibility)
    {
        $groupedResults = [];

        foreach ($regionalDataAvailibility as $result) {
            $indicatorNumber = $result->indicator_number;
            $availabilityLevel = $result->availability_level;

            if (!isset($groupedResults[$indicatorNumber])) {
                $groupedResults[$indicatorNumber] = [
                    'indicatorName' => $result->indicator_short_name,
                    'dataPoints' => [],
                ];
            }

            if (!isset($groupedResults[$indicatorNumber]['dataPoints'][$availabilityLevel])) {
                $groupedResults[$indicatorNumber]['dataPoints'][$availabilityLevel] = 0;
            }
            $groupedResults[$indicatorNumber]['dataPoints'][$availabilityLevel]++;
        }


        $customOrder = [
            'Sufficient' => 1,
            'Insufficient' => 2,
            'No Data' => 3,
            'Not Applicable' => 4
        ];

        $transformedResults = [];
        foreach ($groupedResults as $indicatorNumber => $data) {
            $numberOfAreas = [];
            foreach ($data['dataPoints'] as $category => $count) {
                $numberOfAreas[] = [
                    'category' => $category,
                    'count' => $count
                ];
            }

            $sufficientCount = 0;
            $insufficientCount = 0;
            $noDataCount = 0;

            foreach ($numberOfAreas as $item) {
                if ($item['category'] === 'Sufficient') {
                    $sufficientCount = $item['count'];
                } elseif ($item['category'] === 'Insufficient') {
                    $insufficientCount = $item['count'];
                } elseif ($item['category'] === 'No Data') {
                    $noDataCount = $item['count'];
                }
            }

            usort($numberOfAreas, function ($a, $b) use ($customOrder) {
                return $customOrder[$a['category']] <=> $customOrder[$b['category']];
            });

            $transformedResults[] = [
                'indicatorNumber' => $indicatorNumber,
                'indicatorName' => $data['indicatorName'],
                // 'numberOfAreas' => $numberOfAreas,
                'sufficientCount' => $sufficientCount,
                'insufficientCount' => $insufficientCount,
                'noDataCount' => $noDataCount
            ];
        }

        return $transformedResults;
    }

    public function calculateAvailabilityPercentages($groupedResults)
    {
        foreach ($groupedResults as $goal => &$availabilityLevels) {
            $total = array_sum($availabilityLevels);

            foreach ($availabilityLevels as $level => $count) {
                $percentageKey = "{$level} Percentage";
                $availabilityLevels[$percentageKey] = round(($count / $total) * 100, 2);
            }
        }

        return $groupedResults;
    }

    public function fetchIndicatorPercentagesDataByAreaGoal($area, $goalNumber)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('sdg_performers_summary_data', 'c')
            ->fields('c')
            ->condition('c.goal_number', $goalNumber)
            ->condition('c.area', $area);
        $results = $query->execute()->fetchAll();

        $goal = $this->goalService->getGoalByGoalNumber($goalNumber);
        $count = count($results);
        $groupedResults = [];
        foreach ($results as $result) {
            $tagValue = $result->indicator_tagz;
            if ($tagValue == '0-Insufficient Data') {
                $tagValue = 'Insufficient';
            } else if ($tagValue == 'Non-applicability') {
                $tagValue = 'NA';
            }

            if (!isset($groupedResults[$tagValue])) {
                $groupedResults[$tagValue] = [];
            }
            $groupedResults[$tagValue][] = $result;
        }
        $indicatorPercentagesOverview = [];
        $indicatorPercentagesOverview['goal_number'] = $goal ? $goal->field_goal_number->value : '';
        //$indicatorPercentagesOverview['goal_data_id'] = $goal ? $goal->label() : '';
        $indicatorPercentagesOverview['goal_name'] = $goal ? $goal->label() : '';
        $indicatorPercentagesOverview['image_url'] = isset($goal->image_url) ? $goal->image_url : null;
        $indicatorPercentagesOverview['star_level'] = $result->star_level;
        foreach ($groupedResults as $groupName => $groupItems) {
            $groupCount = count($groupItems);
            $indicatorPercentagesOverview[$groupName] = ($groupCount / $count) * 100;
        }
        return $indicatorPercentagesOverview;
    }

    public function getNationalAnalysisOverviewPerformanceSummaryData($indicatorPercentagesOverview)
    {
        $maxPerformer = -INF;
        $maxRegressing = -INF;
        $topPerformers = [];
        $topRegressings = [];
        $performanceSummary = [];
        //find highest performer and highest regressing
        foreach ($indicatorPercentagesOverview as $item) {
            if (isset($item['Performer']) && $item['Performer'] > $maxPerformer) {
                $maxPerformer = $item['Performer'];
                $topPerformers = [$item];
            } elseif (isset($item['Performer']) && $item['Performer'] == $maxPerformer) {
                $topPerformers[] = $item;
            }

            if (isset($item['Regressing']) && $item['Regressing'] > $maxRegressing) {
                $maxRegressing = $item['Regressing'];
                $topRegressings = [$item];
            } elseif (isset($item['Regressing']) && $item['Regressing'] == $maxRegressing) {
                $topRegressings[] = $item;
            }
        }
        $performanceSummary['best_performers'] = $topPerformers;
        $performanceSummary['worst_performers'] = $topRegressings;
        return $performanceSummary;
    }

    public function fetchIndicatorTrendsDataByAreaGoal($area, $goalNumber)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('sdg_performers_data', 'c')
            ->fields('c')
            ->condition('c.goal_number', $goalNumber)
            ->condition('c.area', $area)
            ->condition('c.series', "NA", "!=");
            
        $sdgPerformersData = $query->execute()->fetchAll();
        $sdgPerformersSummaryData = $this->getSDGPerformersSummaryData();
        foreach ($sdgPerformersData as $result) {
            $result->indicator_tagz = $this->groupIndicatorTrendDataByIndicatorNumber($sdgPerformersSummaryData, $area, $result->ind_number);
        }
        return $sdgPerformersData;
    }

    public function getSDGPerformersSummaryData()
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('sdg_performers_summary_data', 'c')
            ->fields('c');
        $results = $query->execute()->fetchAll();

        return $results;
    }

    public function groupIndicatorTrendDataByIndicatorNumber($sdgPerformersSummaryData, $area, $indicatorNumber)
    {
        $filteredResults = array_filter($sdgPerformersSummaryData, function ($item) use ($area, $indicatorNumber) {
            return $item->ind_number === $indicatorNumber && $item->area === $area;
        });
        if (!empty($filteredResults)) {
            $firstResult = reset($filteredResults);
            return $firstResult->indicator_tagz;
        }
        return null;
    }


    public function groupIndicatorTrendsDataByTarget($indicatorData)
    {
        $groupedResults = [];

        foreach ($indicatorData as $result) {
            $filteredResult = (object) array_filter((array) $result, function ($key) {
                // Exclude properties from retrieved data
                $excludeProperties = [
                    'goal_number',
                    'star_level',
                    'goal_comparison',
                    'direction',
                    'tagz_number',
                    'adjusted_growth_rate',
                    'average_max_year_value',
                    'short_name'
                ];
                return !in_array($key, $excludeProperties);
            }, ARRAY_FILTER_USE_KEY);

            $tagValue = $result->indicator_tagz;
            if ($tagValue === '0-Insufficient Data') {
                $tagValue = 'Insufficient Data';
            }
            if ($tagValue === 'Performer') {
                $tagValue = 'Performing';
            }
            $indicatorNumber = $result->ind_number;
            $seriesNumber = $result->series;
            $seriesName = $result->series_name;
            $seriesTag = $result->tagz;
            if ($seriesTag === "Performer" || $seriesTag === "Top Performer") {
                $seriesTag = "Performing";
            }

            $unitName = $result->unit_name;
            $seriesLevel = $result->series_level;
            $trend = $result->tagz;
            if ($trend === '0-Insufficient Data') {
                $trend = 'Insufficient Data';
            }
            $year = $result->year;
            $dataValue = (float) $result->data_value;

            if (!isset($groupedResults[$tagValue])) {
                $groupedResults[$tagValue] = [];
            }

            $groupedResults[$tagValue][] = [
                'ind_number' => $indicatorNumber,
                'ind_short_name' => $filteredResult->ind_short_name,
                'target_number' => $filteredResult->target_number,
                'area' => $filteredResult->area,
                'trend' => $tagValue,
                'series' => $seriesNumber,
                'series_name' => $seriesName,
                'series_tag' => $seriesTag,
                'unit_name' => $unitName,
                'series_level' => $seriesLevel,
                'year' => $year,
                'data_value' => $dataValue,
            ];
        }

        // Final structure of the response
        $categoryOrder = [
            'Performing' => 1,
            'Stagnant' => 2,
            'Regressing' => 3,
            'Insufficient Data' => 4,
            'Non-applicability' => 5
        ];

        $orderedResults = [];
        foreach ($categoryOrder as $category => $order) {
            if (isset($groupedResults[$category])) {
                $orderedResults[] = [
                    'category' => $category,
                    'data' => $groupedResults[$category],
                ];
            }
        }

        return $orderedResults;
    }

    public function fetchNationalPrioritiesDataByAreaGoal($area, $goalNumber)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('sdg_performers_data', 'c')
            ->fields('c', [
                'sdg_performers_data_id',
                'ind_number',
                'ind_short_name',
                'series_name',
                'area',
                'unit_name',
                'series',
                'series_level',
                'tagz',
                'year',
                'data_value',
            ])
            ->condition('c.goal_number', $goalNumber)
            ->condition('c.area', $area)
            ->condition('c.tagz', 'No Data', '!=');
        
        $results = $query->execute()->fetchAll();

        $results = array_map(function($result) {
            if ($result->tagz == "Performer") {
                $result->tagz = "Performing";
            }
            return $result;
        }, $results);

        $predefinedTrends = [
            'Top Performer',
            'Performing',
            'Stagnant',
            'Regressing',
            'Insufficient Data'
        ];

        $groupedResults = array_map(function($trend) {
            return [
                'category_name' => $trend,
                'data' => []
            ];
        }, $predefinedTrends);

        foreach ($results as $result) {
            foreach ($groupedResults as &$category) {
                if ($category['category_name'] === $result->tagz) {
                    $category['data'][] = (object)[
                        'sdg_performers_data_id' => $result->sdg_performers_data_id,
                        'ind_number' => $result->ind_number,
                        'ind_short_name' => $result->ind_short_name,
                        'series_name' => $result->series_name,
                        'series' => $result->series,
                        'area' => $result->area,
                        'unit_name' => $result->unit_name,
                        'series_level' => $result->series_level,
                        'trend' => $result->tagz,
                        'year' => $result->year,
                        'data_value' => $result->data_value
                    ];
                    break;
                }
            }
        }

        foreach ($groupedResults as &$category) {
            if (!empty($category['data'])) {
                usort($category['data'], function($a, $b) {
                    //     sort ind_number (ascending)
                    $indComparison = strcmp($a->ind_number, $b->ind_number);
                    if ($indComparison !== 0) {
                        return $indComparison;
                    }
    
                    //    sort by year (descending)
                    return strcmp($b->year, $a->year);
                });
            }
        }

        $filteredResults = array_filter($groupedResults, function($category) {
            return !empty($category['data']);
        });
    
        return array_values($filteredResults);
    }

    public function fetchNationalAnalysisDataAvailabilityOverviewData($area, $year)
    {
        $goals = $this->goalService->getAllGoals();
        foreach ($goals as $goalItem) {
            $database = Database::getConnection('default', 'custom');
            $query = $database->select('data_availability_indicator', 'c')
                ->fields('c')
                ->condition('c.country_group', $area)
                // non-repeating indicator will have `repeat` column == 'NA'
                ->condition('c.repeat', 'NA');

            if ($year) {
                $query = $query->condition('c.year', $year);
            }
        }
        $results = $query->execute()->fetchAll();
        return $results;
    }

    public function groupYearlyNationalAnalysisOverviewDataByIndicators($overviewData)
    {
        $groupedResults = [];
        $total = count($overviewData);
    
        foreach ($overviewData as $dataPoint) {
            $year = $dataPoint->year;
            $availabilityLevel = ($year > 2022) ? $dataPoint->availability_level_2015 : $dataPoint->availability_level;
    
            if (!isset($groupedResults[$availabilityLevel])) {
                $groupedResults[$availabilityLevel] = [
                    'name' => $availabilityLevel,
                    'count' => 0,
                ];
            }

            $groupedResults[$availabilityLevel]['count']++;
        }
    
        // mapping grouped results based on predefined order
        $orderAndNames = [
            ['name' => 'Sufficient', 'order' => 1],
            ['name' => 'Insufficient', 'order' => 2],
            ['name' => 'No Data', 'order' => 3],
            ['name' => 'Not Applicable', 'order' => 4],
        ];
    
        usort($groupedResults, function ($a, $b) use ($orderAndNames) {
            $orderA = array_values(array_column($orderAndNames, 'order'))[array_search($a['name'], array_column($orderAndNames, 'name'))];
            $orderB = array_values(array_column($orderAndNames, 'order'))[array_search($b['name'], array_column($orderAndNames, 'name'))];
            return $orderA - $orderB;
        });
    
        $breakdown = [];
        foreach ($groupedResults as $result) {
            $percentage = ($total > 0) ? round(($result['count'] / $total) * 100, 2) : 0;
    
            $breakdown[] = [
                'name' => $result['name'],
                'count' => $result['count'],
                'percentage' => $percentage,
            ];
        }
    
        $breakdown[] = [
            'name' => 'Total',
            'count' => $total,
            'percentage' => 100,
        ];
    
        return ['breakdown' => $breakdown];
    }

    public function groupNationalAnalysisOverviewDataByGoal($overviewData)
    {
        $groupedResults = [];

        foreach ($overviewData as $dataPoint) {
            $goalNumber = $dataPoint->goal;
            sscanf($goalNumber, "Goal %d", $goalNumber);
            $goal = $this->goalService->getGoalByGoalNumber($goalNumber);
            $year = $dataPoint->year;
            $availabilityLevel = ($year > 2022) ? $dataPoint->availability_level_2015 : $dataPoint->availability_level;

            $goalIndex = intval($goalNumber); // Ensure goalNumber is an integer index
            if (!isset($groupedResults[$goalIndex])) {
                $groupedResults[$goalIndex] = [
                    'goal_number' => $goalNumber,
                    'goal_name' => $goal ? $goal->label() : '',
                    'image' => isset($goal->image_url) ? $goal->image_url : 'default_image_url.png', // Default if no image
                    'dataAvailability' => [],
                ];
            }

            if (!isset($groupedResults[$goalIndex]['dataAvailability'][$availabilityLevel])) {
                $groupedResults[$goalIndex]['dataAvailability'][$availabilityLevel] = [
                    'count' => 0,
                ];
            }

            $groupedResults[$goalIndex]['dataAvailability'][$availabilityLevel]['count']++;
        }

        // Calculate percentages and flatten results
        $flattenedResults = [];
        foreach ($groupedResults as $goal) {
            $total = array_sum(array_column($goal['dataAvailability'], 'count'));
            foreach ($goal['dataAvailability'] as $category => $data) {
                $flattenedResults[] = [
                    'image' => $goal['image'],
                    'goal_number' => $goal['goal_number'],
                    'goal_name' => $goal['goal_name'],
                    'category' => $category,
                    'count' => $data['count'],
                    'percentage' => round(($data['count'] / $total) * 100, 2),
                ];
            }
        }

        // Sort flattened results by goal_number
        usort($flattenedResults, function ($a, $b) {
            return $a['goal_number'] - $b['goal_number'];
        });

        return $flattenedResults;
    }


    public function calculateNationalDataAvailabilityPercentages($groupedResults)
    {
        $total = $groupedResults['total'];
        unset($groupedResults['total']);
        $formattedBreakdown = [];
        foreach ($groupedResults['breakdown'] as $name => $data) {
            $count = $data['count'];
            $percentage = round(($count / $total) * 100, 2);

            $formattedBreakdown[] = [
                'name' => $name,
                'count' => $count,
                'percentage' => $percentage,
            ];
        }
        $groupedResults['breakdown'] = $formattedBreakdown;
        $groupedResults['total'] = $total;
        return $groupedResults;
    }

    public function fetchNationalIndicatorDetailsDataAvailabilityByRegionGoal($countryGroup, $goal, $year)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('data_availability_indicator', 'c')
            ->fields('c')
            ->condition('c.country_group', $countryGroup)
            ->condition('c.goal', $goal);

        if ($year) {
            $query->condition('c.year', $year);
        }
        $result = $query->execute()->fetchAll();
        return $result;
    }

    public function groupNationalIndicatorDetailsDataAvailabilityByIndicator($nationalDataAvailabilityIndicators, $country, $year)
    {
        $countryGroup = $this->regionService->getCountryGroupFromCountry($country);
        $associatedCountriesIso = array_column($countryGroup['countries'], 'isoCode');
        $dataSources = $this->fetchDataSources();
        $groupedResults = [];

        foreach ($nationalDataAvailabilityIndicators as $result) {
            $indicatorNumber = $result->indicator_number;
            $availabilityLevel = $result->availability_level_2015;
            $latestYear = $result->latest_year;
            if ($latestYear === NAN) {
                $latestYear = '-';
            }
            if (!isset($groupedResults[$availabilityLevel])) {
                $groupedResults[$availabilityLevel] = [];
            }

            $regionData = $this->fetchRegionalIndicatorDetailsDataAvailabilityByIndicator($indicatorNumber, $year);
            $dataPoints = [];

            foreach ($associatedCountriesIso as $countryIso) {
                foreach ($regionData as $dataPoint) {
                    if ($dataPoint->country_group == $countryIso) {
                        $dataPoints[] = $dataPoint;
                    }
                }
            }

            $regionalDataAvailabilityCount = $this->countRegionalIndicatorDetailsDataAvailabilityByIndicator($dataPoints);
            $filtered = array_filter($dataSources, function ($item) use ($indicatorNumber) {
                return $item->indicator_number === $indicatorNumber;
            });
            $dataSource = reset($filtered);
            if ($dataSource === false) {
                $dataSource = (object) [
                    'compiler' => 'Unknown',
                    'source_category' => 'Unknown'
                ];
            }

            $sufficientCount = 0;
            $insufficientCount = 0;
            $noDataCount = 0;

            foreach ($regionalDataAvailabilityCount as $item) {
                if ($item['category'] === 'Sufficient') {
                    $sufficientCount = $item['count'];
                } elseif ($item['category'] === 'Insufficient') {
                    $insufficientCount = $item['count'];
                } elseif ($item['category'] === 'No Data') {
                    $noDataCount = $item['count'];
                }
            }

            // non-flattened:
            $groupedResults[$availabilityLevel][] = [
                'indicatorNumber' => $result->indicator_number,
                'indicatorName' => $result->indicator_short_name,
                'latestYear' => $latestYear,
                'compiler' => $dataSource->compiler,
                'dataSource' => $dataSource->source_category,
                // 'numberOfAreas' => $regionalDataAvailabilityCount,
                'sufficientCount' => $sufficientCount,
                'insufficientCount' => $insufficientCount,
                'noDataCount' => $noDataCount,
            ];            

        }

        $customOrder = [
            'Sufficient' => 1,
            'Insufficient' => 2,
            'No Data' => 3,
            'Not Applicable' => 4
        ];
        $finalResults = [];
        foreach ($customOrder as $category => $order) {
            if (isset($groupedResults[$category])) {
                $finalResults[] = [
                    'category' => $category,
                    'data' => $groupedResults[$category],
                ];
            }
        }

        return $finalResults;
    }

    public function fetchDataSourceByIndicator($indicatorNumber)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('data_source', 'c')
            ->fields('c')
            ->condition('c.indicator_number', $indicatorNumber);

        $result = $query->execute()->fetchAssoc();
        return $result;
    }

    public function fetchDataSources()
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('data_source', 'c')
            ->fields('c');

        $result = $query->execute()->fetchAll();
        return $result;
    }

    public function fetchRegionalIndicatorDetailsDataAvailabilityByIndicator($indicatorNumber, $year)
    {

        $database = Database::getConnection('default', 'custom');
        $results = [];

        $query = $database->select('data_availability_indicator', 'c')
            ->fields('c')
            ->condition('c.indicator_number', $indicatorNumber);

        if ($year) {
            $query->condition('c.year', $year);
        }

        $result = $query->execute()->fetchAll();
        $results = array_merge($results, $result);

        return $results;
    }

    public function countRegionalIndicatorDetailsDataAvailabilityByIndicator($regionDataDetails)
    {
        $customOrder = [
            'Sufficient' => 1,
            'Insufficient' => 2,
            'No Data' => 3,
            'Not Applicable' => 4
        ];
        $groupedResults = [];
        foreach ($regionDataDetails as $dataPoint) {
            $availabilityLevel = $dataPoint->availability_level_2015;
            if (!isset($groupedResults[$availabilityLevel])) {
                $groupedResults[$availabilityLevel] = 0;
            }
            $groupedResults[$availabilityLevel]++;
        }
        $finalResults = [];
        foreach ($customOrder as $category => $order) {
            if (isset($groupedResults[$category]) && $groupedResults[$category] > 0) {
                $finalResults[] = [
                    'category' => $category,
                    'count' => $groupedResults[$category],
                ];
            }
        }
        return $finalResults;
    }

    public function groupDataAvailabilityIndicatorDetailsByIndicator($indicatorDataDetails)
    {
        $groupedResults = [];
        foreach ($indicatorDataDetails as $dataPoint) {
            $indicatorTagz = $dataPoint->indicator_tagz;

            if (!isset($groupedResults[$indicatorTagz])) {
                $groupedResults[$indicatorTagz] = [];
            }
        }
        return $groupedResults;
    }

    public function fetchCountryGoalProgressData($countryCode, $year)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('country_goal_snapshot_data', 'c')
            ->fields('c')
            ->condition('c.country_group', $countryCode)
            ->condition('c.year', $year);

        $result = $query->execute()->fetchAll(\PDO::FETCH_CLASS, CountryGoalSnapshotData::class);
        return $result;
    }

    public function fetchLatestCountryGoalDataAvailabilityYear($countryGroup)
    {
        $database = Database::getConnection('default', 'custom');
        $query = $database->select('country_goal_snapshot_data', 'c')
            ->fields('c', ['year'])
            ->condition('c.country_group', $countryGroup)
            ->orderBy('c.year', 'DESC')
            ->range(0, 1);
        $highestYear = $query->execute()->fetchField();
        return $highestYear;
    }

    private function compareTargetNumbers($a, $b)
    {
        //this returns a flag on whether or not the target or indicator number that is
        //compared is higher than the other for sorting purposes.
        $a_parts = explode('.', $a);
        $b_parts = explode('.', $b);

        foreach ($a_parts as $index => $a_part) {
            $b_part = isset($b_parts[$index]) ? $b_parts[$index] : '';
            if (is_numeric($a_part) && is_numeric($b_part)) {
                $result = (int) $a_part <=> (int) $b_part;
            } elseif (is_numeric($a_part)) {
                $result = -1;
            } elseif (is_numeric($b_part)) {
                $result = 1;
            } else {
                $result = strcmp($a_part, $b_part);
            }

            if ($result !== 0) {
                return $result;
            }
        }
        return 0;
    }
}
