import { D3BrushEvent } from 'd3-brush';
import { D3DragEvent } from 'd3-drag';
import { D3ZoomEvent } from 'd3-zoom';
import { ComponentConfigInterface } from "../../core/component/config";
import { TrimMode } from "../../types/text";
import { GraphInputLink, GraphInputNode } from "../../types/graph";
import { BooleanAccessor, ColorAccessor, NumericAccessor, StringAccessor, GenericAccessor } from "../../types/accessor";
import { GraphLayoutType, GraphCircleLabel, GraphLinkStyle, GraphLinkArrowStyle, GraphPanelConfig, GraphForceLayoutSettings, GraphElkLayoutSettings, GraphNodeShape, GraphDagreLayoutSetting, GraphNode, GraphLink } from './types';
export interface GraphConfigInterface<N extends GraphInputNode, L extends GraphInputLink> extends ComponentConfigInterface {
    /** Zoom level constraints. Default: [0.35, 1.25] */
    zoomScaleExtent?: [number, number];
    /** Disable zooming. Default: `false` */
    disableZoom?: boolean;
    /** Disable node dragging. Default: `false` */
    disableDrag?: boolean;
    /** Disable brush for multiple node selection. Default: `false` */
    disableBrush?: boolean;
    /** Interval to re-render the graph when zooming. Default: `100` */
    zoomThrottledUpdateNodeThreshold?: number;
    /** Type of the graph layout. Default: `GraphLayoutType.Force` */
    layoutType?: GraphLayoutType | string;
    /** Fit the graph to container on data or config updates, or on container resize. Default: `true` */
    layoutAutofit?: boolean;
    /** Tolerance constant defining whether the graph should be fitted to container
     * (on data or config update, or container resize) after a zoom / pan interaction or not.
     * `0` — Stop fitting after any pan or zoom
     * `Number.POSITIVE_INFINITY` — Always fit
     * Default: `8.0` */
    layoutAutofitTolerance?: number;
    /** Place non-connected nodes at the bottom of the graph. Default: `false` */
    layoutNonConnectedAside?: boolean;
    /** Node group accessor function.
     * Only for `GraphLayoutType.Parallel`, `GraphLayoutType.ParallelHorizontal` and `GraphLayoutType.Concentric` layouts.
     * Default: `node => node.group` */
    layoutNodeGroup?: StringAccessor<N>;
    /** Order of the layout groups.
     * Only for `GraphLayoutType.Parallel`, `GraphLayoutType.ParallelHorizontal` and `GraphLayoutType.Concentric` layouts.
     * Default: `[]` */
    layoutGroupOrder?: string[];
    /** Sets the number of nodes in a sub-group after which they'll continue on the next column (or row if `layoutType` is
     * `GraphLayoutType.ParallelHorizontal`).
     * Only for `GraphLayoutType.Parallel` and `GraphLayoutType.ParallelHorizontal` layouts.
     * Default: `6` */
    layoutParallelNodesPerColumn?: number;
    /** Node sub-group accessor function.
     * Only for `GraphLayoutType.Parallel` and `GraphLayoutType.ParallelHorizontal` layouts.
     * Default: `node => node.subgroup` */
    layoutParallelNodeSubGroup?: StringAccessor<N>;
    /** Number of sub-groups per row (or column if `layoutType` is `GraphLayoutType.ParallelHorizontal`) in a group.
     * Only for `GraphLayoutType.Parallel` and `GraphLayoutType.ParallelHorizontal` layouts.
     * Default: `1` */
    layoutParallelSubGroupsPerRow?: number;
    /** Spacing between groups.
     * Only for `GraphLayoutType.Parallel` and `GraphLayoutType.ParallelHorizontal` layouts.
     * Default: `undefined` */
    layoutParallelGroupSpacing?: number;
    /** Set a group by name to have priority in sorting the graph links.
     * Only for `GraphLayoutType.Parallel` and `GraphLayoutType.ParallelHorizontal` layouts.
     * Default: `undefined` */
    layoutParallelSortConnectionsByGroup?: string;
    /** Force Layout settings, see the `d3-force` package for more details */
    forceLayoutSettings?: GraphForceLayoutSettings<N, L>;
    /** Darge Layout settings, see the `dagrejs` package
     * for more details: https://github.com/dagrejs/dagre/wiki#configuring-the-layout
    */
    dagreLayoutSettings?: GraphDagreLayoutSetting;
    /** ELK layout options, see the `elkjs` package for more details: https://github.com/kieler/elkjs.
     * If you want to specify custom layout option for each node group, you can provide an accessor function that
     * receives group name ('root' for the top-level configuration) as the first argument and returns an object containing
     * layout options. Default: `undefined`
    */
    layoutElkSettings?: GenericAccessor<GraphElkLayoutSettings, string> | undefined;
    /** Array of accessor functions to define nested node groups for the ELK Layered layout.
     * E.g.: `[n => n.group, n => n.subGroup]`.
     * Default: `undefined` */
    layoutElkNodeGroups?: StringAccessor<N>[];
    /** Link width accessor function ot constant value. Default: `1` */
    linkWidth?: NumericAccessor<L>;
    /** Link style accessor function or constant value. Default: `GraphLinkStyle.Solid`  */
    linkStyle?: GenericAccessor<GraphLinkStyle, L>;
    /** Link band width accessor function or constant value. Default: `0` */
    linkBandWidth?: NumericAccessor<L>;
    /** Link arrow accessor function or constant value. Default: `undefined` */
    linkArrow?: GenericAccessor<GraphLinkArrowStyle | string | boolean, L> | undefined;
    /** Link stroke color accessor function or constant value. Default: `undefined` */
    linkStroke?: ColorAccessor<L>;
    /** Link disabled state accessor function or constant value. Default: `false` */
    linkDisabled?: BooleanAccessor<L>;
    /** Link flow animation accessor function or constant value. Default: `false` */
    linkFlow?: BooleanAccessor<L>;
    /** Animation duration of the flow (traffic) circles. Default: `20000` */
    linkFlowAnimDuration?: number;
    /** Size of the moving particles that represent traffic flow. Default: `2` */
    linkFlowParticleSize?: number;
    /** Link label accessor function or constant value. Default: `undefined` */
    linkLabel?: GenericAccessor<GraphCircleLabel, L> | undefined;
    /** Shift label along the link center a little bit to avoid overlap with the link arrow. Default: `true` */
    linkLabelShiftFromCenter?: BooleanAccessor<L>;
    /** Spacing between neighboring links. Default: `8` */
    linkNeighborSpacing?: number;
    /** Curvature of the link. Recommended value range: [0:1.5].
     * `0` - straight line,
     * `1` - nice curvature,
     * `1.5` - very curve.
     * Default: `0` */
    linkCurvature?: NumericAccessor<L>;
    /** Set selected link by its unique id. Default: `undefined` */
    selectedLinkId?: number | string;
    /** Node size accessor function or constant value. Default: `30` */
    nodeSize?: NumericAccessor<N>;
    /** Node stroke width accessor function or constant value. Default: `3` */
    nodeStrokeWidth?: NumericAccessor<N>;
    /** Node shape accessor function or constant value. Default: `GraphNodeShape.Circle` */
    nodeShape?: GenericAccessor<GraphNodeShape | string, N>;
    /** Node gauge outline accessor function or constant value in the range [0,100]. Default: `0` */
    nodeGaugeValue?: NumericAccessor<N>;
    /** Node gauge outline fill color accessor function or constant value. Default: `undefined` */
    nodeGaugeFill?: ColorAccessor<N>;
    /** Animation duration of the node gauge outline. Default: `1500` */
    nodeGaugeAnimDuration?: number;
    /** Node central icon accessor function or constant value. Default: `node => node.icon` */
    nodeIcon?: StringAccessor<N>;
    /** Node central icon size accessor function or constant value. Default: `undefined` */
    nodeIconSize?: NumericAccessor<N>;
    /** Node label accessor function or constant value. Default: `node => node.label` */
    nodeLabel?: StringAccessor<N>;
    /** Defines whether to trim the node labels or not. Default: `true` */
    nodeLabelTrim?: BooleanAccessor<N>;
    /** Node label trimming mode. Default: `TrimMode.Middle` */
    nodeLabelTrimMode?: GenericAccessor<TrimMode | string, N>;
    /** Node label maximum allowed text length above which the label will be trimmed. Default: `15` */
    nodeLabelTrimLength?: NumericAccessor<N>;
    /** Node sub-label accessor function or constant value: Default: `''` */
    nodeSubLabel?: StringAccessor<N>;
    /** Defines whether to trim the node sub-labels or not. Default: `true` */
    nodeSubLabelTrim?: BooleanAccessor<N>;
    /** Node sub-label trimming mode. Default: `TrimMode.Middle` */
    nodeSubLabelTrimMode?: GenericAccessor<TrimMode | string, N>;
    /** Node sub-label maximum allowed text length above which the label will be trimmed. Default: `15` */
    nodeSubLabelTrimLength?: NumericAccessor<N>;
    /** Node circular side labels accessor function. The function should return an array of GraphCircleLabel objects. Default: `undefined` */
    nodeSideLabels?: GenericAccessor<GraphCircleLabel[], N>;
    /** Node bottom icon accessor function. Default: `undefined` */
    nodeBottomIcon?: StringAccessor<N>;
    /** Node disabled state accessor function or constant value. Default: `false` */
    nodeDisabled?: BooleanAccessor<N>;
    /** Node fill color accessor function or constant value. Default: `node => node.fill` */
    nodeFill?: ColorAccessor<N>;
    /** Node stroke color accessor function or constant value. Default: `node => node.stroke` */
    nodeStroke?: ColorAccessor<N>;
    /** Sorting function to determine node placement. Default: `undefined` */
    nodeSort?: ((a: N, b: N) => number);
    /** Specify the initial position for entering nodes as [x, y]. Default: `undefined` */
    nodeEnterPosition?: GenericAccessor<[number, number], N> | undefined;
    /** Specify the initial scale for entering nodes in the range [0,1]. Default: `0.75` */
    nodeEnterScale?: NumericAccessor<N> | undefined;
    /** Specify the destination position for exiting nodes as [x, y]. Default: `undefined` */
    nodeExitPosition?: GenericAccessor<[number, number], N> | undefined;
    /** Specify the destination scale for exiting nodes in the range [0,1]. Default: `0.75` */
    nodeExitScale?: NumericAccessor<N> | undefined;
    /** Set selected node by unique id. Default: `undefined` */
    selectedNodeId?: number | string;
    /** Set selected nodes by unique id. Default: `undefined` */
    selectedNodeIds?: number[] | string[];
    /** Panels configuration. An array of `GraphPanelConfig` objects. Default: `[]` */
    panels?: GraphPanelConfig[] | undefined;
    /** Graph node drag start callback function. Default: `undefined` */
    onNodeDragStart?: (n: GraphNode<N, L>, event: D3DragEvent<SVGGElement, GraphNode<N, L>, unknown>) => void | undefined;
    /** Graph node drag callback function. Default: `undefined` */
    onNodeDrag?: (n: GraphNode<N, L>, event: D3DragEvent<SVGGElement, GraphNode<N, L>, unknown>) => void | undefined;
    /** Graph node drag end callback function. Default: `undefined` */
    onNodeDragEnd?: (n: GraphNode<N, L>, event: D3DragEvent<SVGGElement, GraphNode<N, L>, unknown>) => void | undefined;
    /** Zoom event callback. Default: `undefined` */
    onZoom?: (zoomScale: number, zoomScaleExtent: [number, number], event: D3ZoomEvent<SVGGElement, unknown> | undefined) => void;
    /** Callback function to be called when the graph layout is calculated. Default: `undefined` */
    onLayoutCalculated?: (n: GraphNode<N, L>[], links: GraphLink<N, L>[]) => void;
    /** Graph node selection brush callback function. Default: `undefined` */
    onNodeSelectionBrush?: (selectedNodes: GraphNode<N, L>[], event: D3BrushEvent<SVGGElement> | undefined) => void;
    /** Graph multiple node drag callback function. Default: `undefined` */
    onNodeSelectionDrag?: (selectedNodes: GraphNode<N, L>[], event: D3DragEvent<SVGGElement, GraphNode<N, L>, unknown>) => void;
}
export declare const GraphDefaultConfig: GraphConfigInterface<GraphInputNode, GraphInputLink>;
