import { select } from 'd3-selection';
import { range } from 'd3-array';
import toPx from 'to-px';
import { getBoolean, getValue, getNumber, throttle } from '../../../../utils/data.js';
import { smartTransition } from '../../../../utils/d3.js';
import { getCSSVariableValueInPixels } from '../../../../utils/misc.js';
import { estimateStringPixelLength } from '../../../../utils/text.js';
import { GraphLinkStyle } from '../../types.js';
import { getX, getY } from '../node/helper.js';
import { getLinkBandWidth, getLinkColor, getLinkShiftTransform, getLinkStrokeWidth, getLinkArrowStyle, LINK_MARKER_WIDTH, getLinkLabelTextColor } from './helper.js';
import { ZoomLevel } from '../zoom-levels.js';
import { zoomOutLevel2 } from '../../style.js';
import { linkSupport, link, linkBand, linkArrow, flowGroup, flowCircle, linkLabelGroup, linkLabelBackground, linkLabelContent, greyedOutLink, linkDashed } from './style.js';

function createLinks(selection) {
    selection.attr('opacity', 0);
    selection.append('path')
        .attr('class', linkSupport);
    selection.append('path')
        .attr('class', link);
    selection.append('path')
        .attr('class', linkBand);
    selection.append('use')
        .attr('class', linkArrow);
    selection.append('g')
        .attr('class', flowGroup)
        .selectAll(`.${flowCircle}`)
        .data(range(0, 6)).enter()
        .append('circle')
        .attr('class', flowCircle);
    const linkLabelGroup$1 = selection.append('g')
        .attr('class', linkLabelGroup);
    linkLabelGroup$1.append('rect')
        .attr('class', linkLabelBackground);
    linkLabelGroup$1.append('text')
        .attr('class', linkLabelContent);
}
function updateSelectedLinks(selection, config, scale) {
    const isGreyedOut = (d, i) => getBoolean(d, config.linkDisabled, i) || d._state.greyout;
    selection
        .classed(greyedOutLink, (d, i) => isGreyedOut(d, i));
    selection.each((d, i, elements) => {
        const element = elements[i];
        const group = select(element);
        group.select(`.${link}`);
        group.select(`.${linkBand}`);
        const linkSupport$1 = group.select(`.${linkSupport}`);
        linkSupport$1
            .style('stroke-opacity', (d._state.hovered || d._state.selected) ? 0.2 : 0)
            .style('stroke-width', d._state.selected
            ? getLinkBandWidth(d, scale, config) + 5
            : d._state.hovered ? getLinkBandWidth(d, scale, config) + 10 : null);
    });
}
function updateLinks(selection, config, duration, scale = 1, getLinkArrowDefId) {
    const { linkFlowParticleSize, linkStyle, linkFlow, linkLabel, linkLabelShiftFromCenter } = config;
    if (!selection.size())
        return;
    selection
        .classed(linkDashed, d => getValue(d, linkStyle, d._indexGlobal) === GraphLinkStyle.Dashed);
    selection.each((d, i, elements) => {
        var _a, _b, _c, _d, _e;
        const element = elements[i];
        const linkGroup = select(element);
        const link$1 = linkGroup.select(`.${link}`);
        const linkBand$1 = linkGroup.select(`.${linkBand}`);
        const linkSupport$1 = linkGroup.select(`.${linkSupport}`);
        const linkArrow$1 = linkGroup.select(`.${linkArrow}`);
        const flowGroup$1 = linkGroup.select(`.${flowGroup}`);
        const linkColor = getLinkColor(d, config);
        const linkShiftTransform = getLinkShiftTransform(d, config.linkNeighborSpacing);
        const linkLabelDatum = getValue(d, linkLabel, d._indexGlobal);
        const linkLabelText = linkLabelDatum ? (_a = linkLabelDatum.text) === null || _a === void 0 ? void 0 : _a.toString() : undefined;
        const x1 = getX(d.source);
        const y1 = getY(d.source);
        const x2 = getX(d.target);
        const y2 = getY(d.target);
        const curvature = (_b = getNumber(d, config.linkCurvature, i)) !== null && _b !== void 0 ? _b : 0;
        const cp1x = x1 + (x2 - x1) * 0.5 * curvature;
        const cp1y = y1 + (y2 - y1) * 0.0 * curvature;
        const cp2x = x1 + (x2 - x1) * 0.5 * curvature;
        const cp2y = y1 + (y2 - y1) * 1.0 * curvature;
        const pathData = `M${x1},${y1} C${cp1x},${cp1y} ${cp2x},${cp2y} ${x2},${y2}`;
        link$1
            .attr('class', link)
            .style('stroke-width', getLinkStrokeWidth(d, scale, config))
            .style('stroke', linkColor)
            .attr('transform', linkShiftTransform);
        smartTransition(link$1, duration)
            .attr('d', pathData);
        linkBand$1
            .attr('class', linkBand)
            .attr('transform', linkShiftTransform)
            .style('stroke-width', getLinkBandWidth(d, scale, config))
            .style('stroke', linkColor);
        smartTransition(linkBand$1, duration)
            .attr('d', pathData);
        linkSupport$1
            .style('stroke', linkColor)
            .attr('transform', linkShiftTransform)
            .attr('d', pathData);
        // Arrow
        const linkArrowStyle = getLinkArrowStyle(d, config);
        const linkPathElement = linkSupport$1.node();
        const pathLength = linkPathElement.getTotalLength();
        if (linkArrowStyle) {
            const arrowPos = pathLength * (linkLabelText ? 0.65 : 0.5);
            const p1 = linkPathElement.getPointAtLength(arrowPos);
            const p2 = linkPathElement.getPointAtLength(arrowPos + 1); // A point very close to p1
            // Calculate the angle for the arrowhead
            const angle = Math.atan2(p2.y - p1.y, p2.x - p1.x) * (180 / Math.PI);
            const arrowWasShownBefore = linkArrow$1.attr('href');
            linkArrow$1
                .attr('href', `#${getLinkArrowDefId(linkArrowStyle)}`);
            smartTransition(linkArrow$1, arrowWasShownBefore ? duration : 0)
                .attr('fill', linkColor)
                .attr('transform', `translate(${p1.x}, ${p1.y}) rotate(${angle})`);
        }
        else {
            linkArrow$1.attr('href', null);
        }
        // Particle Flow
        flowGroup$1
            .attr('transform', linkShiftTransform)
            .style('display', getBoolean(d, linkFlow, d._indexGlobal) ? null : 'none')
            .style('opacity', 0);
        flowGroup$1
            .selectAll(`.${flowCircle}`)
            .attr('r', linkFlowParticleSize / scale)
            .style('fill', linkColor);
        smartTransition(flowGroup$1, duration)
            .style('opacity', scale < ZoomLevel.Level2 ? 0 : 1);
        // Labels
        const linkLabelGroup$1 = linkGroup.select(`.${linkLabelGroup}`);
        if (linkLabelText) {
            const linkMarkerWidth = linkArrowStyle ? LINK_MARKER_WIDTH * 2 : 0;
            const linkLabelShift = getBoolean(d, linkLabelShiftFromCenter, d._indexGlobal) ? -linkMarkerWidth + 4 : 0;
            const linkLabelPos = linkPathElement.getPointAtLength(pathLength / 2 + linkLabelShift);
            const linkLabelTranslate = `translate(${linkLabelPos.x}, ${linkLabelPos.y})`;
            const linkLabelBackground$1 = linkLabelGroup$1.select(`.${linkLabelBackground}`);
            const linkLabelContent$1 = linkLabelGroup$1.select(`.${linkLabelContent}`);
            // If the label was hidden or didn't have text before, we need to set the initial position
            if (!linkLabelContent$1.text() || linkLabelContent$1.attr('hidden')) {
                linkLabelGroup$1.attr('transform', linkLabelTranslate);
            }
            linkLabelGroup$1.attr('hidden', null)
                .style('cursor', linkLabelDatum.cursor);
            smartTransition(linkLabelGroup$1, duration)
                .attr('transform', linkLabelTranslate)
                .style('opacity', 1);
            linkLabelContent$1
                .text(linkLabelText)
                .attr('dy', '0.1em')
                .style('font-size', linkLabelDatum.fontSize)
                .style('fill', (_c = linkLabelDatum.textColor) !== null && _c !== void 0 ? _c : getLinkLabelTextColor(linkLabelDatum));
            const shouldBeRenderedAsCircle = linkLabelText.length <= 2;
            const linkLabelPaddingVertical = 4;
            const linkLabelPaddingHorizontal = shouldBeRenderedAsCircle ? linkLabelPaddingVertical : 8;
            const linkLabelFontSize = (_d = toPx(linkLabelDatum.fontSize)) !== null && _d !== void 0 ? _d : getCSSVariableValueInPixels('var(--vis-graph-link-label-font-size)', linkLabelContent$1.node());
            const linkLabelWidthPx = estimateStringPixelLength(linkLabelText, linkLabelFontSize);
            const linkLabelBackgroundBorderRadius = (_e = linkLabelDatum.radius) !== null && _e !== void 0 ? _e : (shouldBeRenderedAsCircle ? linkLabelFontSize : 4);
            const linkLabelBackgroundWidth = (shouldBeRenderedAsCircle ? linkLabelFontSize : linkLabelWidthPx);
            linkLabelBackground$1
                .attr('x', -linkLabelBackgroundWidth / 2 - linkLabelPaddingHorizontal)
                .attr('y', -linkLabelFontSize / 2 - linkLabelPaddingVertical)
                .attr('width', linkLabelBackgroundWidth + linkLabelPaddingHorizontal * 2)
                .attr('height', linkLabelFontSize + linkLabelPaddingVertical * 2)
                .attr('rx', linkLabelBackgroundBorderRadius)
                .style('fill', linkLabelDatum.color);
        }
        else {
            linkLabelGroup$1.attr('hidden', true);
        }
    });
    // Pointer Events
    if (duration > 0) {
        selection.attr('pointer-events', 'none');
        const t = smartTransition(selection, duration);
        t
            .attr('opacity', 1)
            .on('end interrupt', (d, i, elements) => {
            select(elements[i])
                .attr('pointer-events', 'stroke')
                .attr('opacity', 1);
        });
    }
    else {
        selection.attr('opacity', 1);
    }
    updateSelectedLinks(selection, config, scale);
}
function removeLinks(selection, config, duration) {
    smartTransition(selection, duration / 2)
        .attr('opacity', 0)
        .remove();
}
function animateLinkFlow(selection, config, scale) {
    const { linkFlow } = config;
    if (scale < ZoomLevel.Level2)
        return;
    selection.each((d, i, elements) => {
        const element = elements[i];
        const linkGroup = select(element);
        const flowGroup$1 = linkGroup.select(`.${flowGroup}`);
        const linkPathElement = linkGroup.select(`.${linkSupport}`).node();
        const pathLength = linkPathElement.getTotalLength();
        if (!getBoolean(d, linkFlow, d._indexGlobal))
            return;
        const t = d._state.flowAnimTime;
        const circles = flowGroup$1.selectAll(`.${flowCircle}`);
        circles
            .attr('transform', index => {
            const tt = (t + (+index) / (circles.size() - 1)) % 1;
            const p = linkPathElement.getPointAtLength(tt * pathLength);
            return `translate(${p.x}, ${p.y})`;
        });
    });
}
function zoomLinks(selection, config, scale) {
    const { linkFlowParticleSize } = config;
    selection.classed(zoomOutLevel2, scale < ZoomLevel.Level2);
    selection.selectAll(`.${flowCircle}`)
        .attr('r', linkFlowParticleSize / scale);
    const linkElements = selection.selectAll(`.${link}`);
    linkElements
        .style('stroke-width', d => getLinkStrokeWidth(d, scale, config));
    const linkBandElements = selection.selectAll(`.${linkBand}`);
    linkBandElements
        .style('stroke-width', d => getLinkBandWidth(d, scale, config));
}
const zoomLinksThrottled = throttle(zoomLinks, 500);

export { animateLinkFlow, createLinks, removeLinks, updateLinks, updateSelectedLinks, zoomLinks, zoomLinksThrottled };
//# sourceMappingURL=index.js.map
