import { select } from 'd3-selection';
import { line } from 'd3-shape';
import { interpolatePath } from 'd3-interpolate-path';
import { XYComponentCore } from '../../core/xy-component/index.js';
import { isNumber, isArray, getNumber, getString, getValue } from '../../utils/data.js';
import { smartTransition } from '../../utils/d3.js';
import { getColor } from '../../utils/color.js';
import { Curve, CurveType } from '../../types/curve.js';
import { Direction } from '../../types/direction.js';
import { LineDefaultConfig } from './config.js';
import * as style from './style.js';
import { line as line$1, linePath, lineSelectionHelper, dim } from './style.js';

class Line extends XYComponentCore {
    constructor(config) {
        super();
        this._defaultConfig = LineDefaultConfig;
        this.config = this._defaultConfig;
        this.curve = Curve[CurveType.MonotoneX];
        this.events = {
            [Line.selectors.line]: {
                mouseover: this._highlight.bind(this),
                mouseleave: this._resetHighlight.bind(this),
            },
        };
        if (config)
            this.setConfig(config);
    }
    get bleed() {
        const { config: { lineWidth } } = this;
        const yDomain = this.yScale.domain();
        const yDirection = this.yScale.range()[0] > this.yScale.range()[1]
            ? Direction.North
            : Direction.South;
        const isYDirectionSouth = yDirection === Direction.South;
        const isLineThick = lineWidth > 3;
        const isLineVeryThick = lineWidth >= 10;
        return {
            top: !isLineVeryThick && ((!isYDirectionSouth && (yDomain[1] === 0)) || (isYDirectionSouth && (yDomain[0] === 0))) ? 0 : lineWidth / 2,
            bottom: !isLineVeryThick && ((!isYDirectionSouth && (yDomain[0] === 0)) || (isYDirectionSouth && (yDomain[1] === 0))) ? 0 : lineWidth / 2,
            left: isLineThick ? lineWidth / 2 : 0,
            right: isLineThick ? lineWidth / 2 : 0,
        };
    }
    _render(customDuration) {
        super._render(customDuration);
        const { config, datamodel: { data } } = this;
        const duration = isNumber(customDuration) ? customDuration : config.duration;
        this.curve = Curve[config.curveType];
        this.lineGen = line()
            .x(d => d.x)
            .y(d => d.y)
            .defined(d => d.defined)
            .curve(this.curve);
        const yAccessors = (isArray(config.y) ? config.y : [config.y]);
        const lineDataX = data.map((d, i) => this.xScale(getNumber(d, config.x, i)));
        const lineData = yAccessors.map(a => {
            const ld = data.map((d, i) => {
                const rawValue = getNumber(d, a, i);
                // If `rawValue` is not numerical or if it's not finite (`NaN`, `undefined`, ...), we replace it with `config.fallbackValue`
                const value = (isNumber(rawValue) || (rawValue === null)) && isFinite(rawValue) ? rawValue : config.fallbackValue;
                return {
                    x: lineDataX[i],
                    y: this.yScale(value !== null && value !== void 0 ? value : 0),
                    defined: isFinite(value),
                    value,
                };
            });
            const defined = ld.reduce((def, d) => (d.defined || def), false);
            // If the line consists only of `null` values, we'll still render it but it'll be invisible.
            // Such trick allows us to have better animated transitions.
            const visible = defined && ld.some(d => d.value !== null);
            return {
                values: ld,
                defined,
                visible,
            };
        });
        const lines = this.g
            .selectAll(`.${line$1}`)
            .data(lineData);
        const linesEnter = lines.enter().append('g')
            .attr('class', line$1);
        linesEnter
            .append('path')
            .attr('class', linePath)
            .attr('stroke', (d, i) => getColor(data, config.color, i))
            .attr('stroke-opacity', 0)
            .attr('stroke-width', config.lineWidth);
        linesEnter
            .append('path')
            .attr('class', lineSelectionHelper)
            .attr('d', this._emptyPath());
        const linesMerged = linesEnter.merge(lines);
        linesMerged.style('cursor', (d, i) => getString(data, config.cursor, i));
        linesMerged.each((d, i, elements) => {
            var _a;
            const group = select(elements[i]);
            const linePath$1 = group.select(`.${linePath}`);
            const lineSelectionHelper$1 = group.select(`.${lineSelectionHelper}`);
            const isLineVisible = d.visible;
            const dashArray = getValue(data, config.lineDashArray, i);
            const transition = smartTransition(linePath$1, duration)
                .attr('stroke', getColor(data, config.color, i))
                .attr('stroke-width', config.lineWidth)
                .attr('stroke-opacity', isLineVisible ? 1 : 0)
                .style('stroke-dasharray', (_a = dashArray === null || dashArray === void 0 ? void 0 : dashArray.join(' ')) !== null && _a !== void 0 ? _a : null); // We use `.style` because there's also a default CSS-variable for stroke-dasharray
            const hasUndefinedSegments = d.values.some(d => !d.defined);
            const svgPathD = this.lineGen(d.values);
            if (duration && !hasUndefinedSegments) {
                const previous = linePath$1.attr('d') || this._emptyPath();
                const next = svgPathD || this._emptyPath();
                const t = transition;
                t.attrTween('d', () => interpolatePath(previous, next));
            }
            else if (d.visible) {
                transition.attr('d', svgPathD);
            }
            lineSelectionHelper$1
                .attr('d', svgPathD)
                .attr('visibility', isLineVisible ? null : 'hidden');
        });
        smartTransition(lines.exit(), duration)
            .style('opacity', 0)
            .remove();
    }
    _emptyPath() {
        const xRange = this.xScale.range();
        const yRange = this.yScale.range();
        return `M${xRange[0]},${yRange[0]} L${xRange[1]},${yRange[0]}`;
    }
    _highlight(datum) {
        const { config } = this;
        if (config.highlightOnHover) {
            this.g
                .selectAll(`.${line$1}`)
                .classed(dim, d => d !== datum);
        }
    }
    _resetHighlight() {
        const { config } = this;
        if (config.highlightOnHover) {
            this.g
                .selectAll(`.${line$1}`)
                .classed(dim, false);
        }
    }
}
Line.selectors = style;

export { Line };
//# sourceMappingURL=index.js.map
