import { select } from 'd3-selection';
import { symbol } from 'd3-shape';
import { color } from 'd3-color';
import { Position } from '../../../types/position.js';
import { Symbol } from '../../../types/symbol.js';
import { smartTransition } from '../../../utils/d3.js';
import { isStringCSSVariable, getCSSVariableValue } from '../../../utils/misc.js';
import { hexToBrightness } from '../../../utils/color.js';
import { getCentralLabelFontSize, getLabelShift } from './utils.js';

function createPoints(selection, xScale, yScale) {
    selection.attr('transform', d => `translate(${d._point.xValue},${d._point.yValue})`);
    selection.append('path').style('fill', d => d._point.color);
    selection.append('text')
        .style('pointer-events', 'none');
    selection.attr('transform', d => `translate(${xScale(d._point.xValue)},${yScale(d._point.yValue)}) scale(0)`);
}
function updatePoints(selection, config, xScale, yScale, duration) {
    const symbolGenerator = symbol();
    selection.each((d, index, elements) => {
        var _a, _b, _c, _d;
        const group = select(elements[index]);
        const label = group.select('text');
        const path = group.select('path');
        // Shape
        const pointDiameter = d._point.sizePx;
        const pointColor = d._point.color;
        const pointStrokeColor = (_a = d._point.strokeColor) !== null && _a !== void 0 ? _a : null;
        const pointStrokeWidth = (_b = d._point.strokeWidthPx) !== null && _b !== void 0 ? _b : null;
        path.attr('d', () => {
            const svgPath = d._point.shape ? symbolGenerator
                .size(Math.PI * pointDiameter * pointDiameter / 4)
                .type(Symbol[d._point.shape])() : null;
            return svgPath;
        });
        smartTransition(path, duration)
            .style('fill', pointColor)
            .style('stroke', pointStrokeColor)
            .style('stroke-width', `${pointStrokeWidth}px`);
        // Label
        const labelPosition = d._point.labelPosition;
        const isLabelPositionCenter = (labelPosition !== Position.Top) && (labelPosition !== Position.Bottom) &&
            (labelPosition !== Position.Left) && (labelPosition !== Position.Right);
        const pointLabelText = (_c = d._point.label) !== null && _c !== void 0 ? _c : '';
        const textLength = pointLabelText.length;
        const centralLabelFontSize = getCentralLabelFontSize(pointDiameter, textLength);
        let labelColor = d._point.labelColor;
        if (!labelColor && isLabelPositionCenter) {
            const c = pointColor || 'var(--vis-scatter-fill-color)';
            const hex = (_d = color(isStringCSSVariable(c) ? getCSSVariableValue(c, group.node()) : c)) === null || _d === void 0 ? void 0 : _d.hex();
            const brightness = hexToBrightness(hex);
            labelColor = brightness > config.labelTextBrightnessRatio ? 'var(--vis-scatter-point-label-text-color-dark)' : 'var(--vis-scatter-point-label-text-color-light)';
        }
        const labelShift = getLabelShift(labelPosition, pointDiameter);
        label.html(pointLabelText)
            .attr('x', labelShift[0])
            .attr('y', labelShift[1])
            .style('font-size', isLabelPositionCenter ? centralLabelFontSize : null)
            .style('text-anchor', () => {
            switch (labelPosition) {
                case Position.Right: return null;
                case Position.Left: return 'end';
                default: return 'middle';
            }
        })
            .style('dominant-baseline', () => {
            switch (labelPosition) {
                case Position.Top: return null;
                case Position.Bottom: return 'hanging';
                default: return 'central';
            }
        });
        smartTransition(label, duration)
            .style('fill', labelColor);
        path.style('cursor', d._point.cursor);
    });
    smartTransition(selection, duration)
        .attr('transform', d => `translate(${xScale(d._point.xValue)},${yScale(d._point.yValue)}) scale(1)`);
}
function removePoints(selection, xScale, yScale, duration) {
    smartTransition(selection, duration)
        .attr('transform', d => `translate(${xScale(d._point.xValue)},${yScale(d._point.yValue)}) scale(0)`)
        .remove();
}

export { createPoints, removePoints, updatePoints };
//# sourceMappingURL=point.js.map
