import { css } from '@emotion/css';
import { extent, merge } from 'd3-array';
import '../../styles/index.js';
import { ContainerCore } from '../../core/container/index.js';
import { CoreDataModel } from '../../data-models/core.js';
import { AxisType } from '../../components/axis/types.js';
import { ScaleDimension } from '../../types/scale.js';
import { Direction } from '../../types/direction.js';
import { clamp, clean, flatten } from '../../utils/data.js';
import { guid } from '../../utils/misc.js';
import { XYContainerDefaultConfig } from './config.js';

class XYContainer extends ContainerCore {
    constructor(element, config, data) {
        var _a, _b;
        super(element);
        this._defaultConfig = XYContainerDefaultConfig;
        this.datamodel = new CoreDataModel();
        this.config = this._defaultConfig;
        this._clipPathId = guid();
        this._axisMargin = { top: 0, bottom: 0, left: 0, right: 0 };
        this._firstRender = true;
        this._clipPath = this.svg.append('clipPath')
            .attr('id', this._clipPathId);
        this._clipPath.append('rect');
        // When the base tag is specified on the HTML head,
        //  Safari fails to find the corresponding filter URL.
        //  We have to provide tull url in order to fix that
        const highlightFilterId = 'saturate';
        const baseUrl = window.location.href.replace(window.location.hash, '');
        this.svg.attr('class', css `
      --highlight-filter-id: url(${baseUrl}#${highlightFilterId}); // defining a css variable
    `);
        this._svgDefs.append('filter')
            .attr('id', highlightFilterId)
            .attr('filterUnits', 'objectBoundingBox')
            .html('<feColorMatrix type="saturate" in="SourceGraphic" values="1.35"/>');
        if (config) {
            this.updateContainer(config, true);
        }
        if (data) {
            this.setData(data, true);
        }
        // Render if there are axes or components with data
        if (this.config.xAxis ||
            this.config.yAxis ||
            ((_a = this.components) === null || _a === void 0 ? void 0 : _a.some(c => c.datamodel.data))) {
            this.render();
        }
        // Force re-render axes when fonts are loaded
        (_b = document.fonts) === null || _b === void 0 ? void 0 : _b.ready.then(() => {
            if (!this._firstRender)
                this._renderAxes(0);
        });
    }
    get components() {
        return this.config.components;
    }
    // Overriding ContainerCore default get width method to work with axis auto margin
    get width() {
        const margin = this._getMargin();
        return clamp(this.containerWidth - margin.left - margin.right, 0, Number.POSITIVE_INFINITY);
    }
    // Overriding ContainerCore default get height method to work with axis auto margin
    get height() {
        const margin = this._getMargin();
        return clamp(this.containerHeight - margin.top - margin.bottom, 0, Number.POSITIVE_INFINITY);
    }
    setData(data, preventRender) {
        var _a, _b, _c, _d;
        const { components, config } = this;
        if (!data)
            return;
        this.datamodel.data = data;
        components.forEach((c) => {
            c.setData(data);
        });
        (_a = config.crosshair) === null || _a === void 0 ? void 0 : _a.setData(data);
        (_b = config.xAxis) === null || _b === void 0 ? void 0 : _b.setData(data);
        (_c = config.yAxis) === null || _c === void 0 ? void 0 : _c.setData(data);
        (_d = config.tooltip) === null || _d === void 0 ? void 0 : _d.hide();
        if (!preventRender)
            this.render();
    }
    updateContainer(containerConfig, preventRender) {
        super.updateContainer(containerConfig);
        this._removeAllChildren();
        // If there were any new components added we need to pass them data
        this.setData(this.datamodel.data, true);
        // Set up the axes
        if (containerConfig.xAxis) {
            this.config.xAxis.config.type = AxisType.X;
            this.element.appendChild(containerConfig.xAxis.element);
        }
        if (containerConfig.yAxis) {
            this.config.yAxis.config.type = AxisType.Y;
            this.element.appendChild(containerConfig.yAxis.element);
        }
        // Re-insert elements to the DOM
        for (const c of this.components) {
            this.element.appendChild(c.element);
        }
        // Set up the tooltip
        const tooltip = containerConfig.tooltip;
        if (tooltip) {
            if (!tooltip.hasContainer())
                tooltip.setContainer(this._container);
            tooltip.setComponents(this.components);
        }
        // Set up the crosshair
        const crosshair = containerConfig.crosshair;
        if (crosshair) {
            crosshair.setContainer(this.svg);
            crosshair.tooltip = tooltip;
            this.element.appendChild(crosshair.element);
        }
        // Set up annotations
        const annotations = containerConfig.annotations;
        if (annotations) {
            this.element.appendChild(annotations.element);
        }
        // Clipping path
        this.element.appendChild(this._clipPath.node());
        // Defs
        this.element.appendChild(this._svgDefs.node());
        this.element.appendChild(this._svgDefsExternal.node());
        // Rendering
        if (!preventRender)
            this.render();
    }
    updateComponents(componentConfigs, preventRender) {
        const { config } = this;
        this.components.forEach((c, i) => {
            const componentConfig = componentConfigs[i];
            if (componentConfig) {
                c.setConfig(componentConfigs[i]);
            }
        });
        this._updateScales(...this.components, config.xAxis, config.yAxis, config.crosshair);
        if (!preventRender)
            this.render();
    }
    update(containerConfig, componentConfigs, data) {
        if (data)
            this.datamodel.data = data; // Just updating the data model because the `updateContainer` method has the `setData` step inside
        if (containerConfig)
            this.updateContainer(containerConfig, true);
        if (componentConfigs)
            this.updateComponents(componentConfigs, true);
        this.render();
    }
    _preRender() {
        const { config } = this;
        super._preRender();
        // Calculate extra margin required to fit the axes
        if (config.autoMargin) {
            this._setAutoMargin();
        }
        // Pass size to the components
        const components = clean([...this.components, config.xAxis, config.yAxis, config.crosshair, config.annotations]);
        for (const c of components) {
            c.setSize(this.width, this.height, this.containerWidth, this.containerHeight);
        }
        // Update Scales of all the components at once to calculate required paddings and sync them
        this._updateScales(...this.components, config.xAxis, config.yAxis, config.crosshair);
    }
    _render(customDuration) {
        var _a, _b, _c, _d, _e;
        const { config } = this;
        super._render();
        // Get chart total margin after auto margin calculations
        const margin = this._getMargin();
        // Render components
        for (const c of this.components) {
            c.g.attr('transform', `translate(${margin.left},${margin.top})`)
                .style('clip-path', c.clippable ? `url(#${this._clipPathId})` : null)
                .style('-webkit-clip-path', c.clippable ? `url(#${this._clipPathId})` : null);
            c.render(customDuration);
        }
        this._renderAxes(this._firstRender ? 0 : customDuration);
        // Clip RectsetSize
        // Extending the clipping path to allow small overflow (e.g. Line will looks better that way when it touches the edges)
        const clipPathExtension = 2;
        this._clipPath.select('rect')
            .attr('x', -clipPathExtension)
            .attr('y', -clipPathExtension)
            .attr('width', this.width + 2 * clipPathExtension)
            .attr('height', this.height + 2 * clipPathExtension);
        // Tooltip
        (_a = config.tooltip) === null || _a === void 0 ? void 0 : _a.update(); // Re-bind events
        // Crosshair
        const crosshair = config.crosshair;
        if (crosshair) {
            // Pass accessors
            const yAccessors = this.components.filter(c => !c.stacked).map(c => c.config.y);
            const yStackedAccessors = this.components.filter(c => c.stacked).map(c => c.config.y);
            const baselineComponentConfig = (_b = this.components.find(c => c.config.baseline)) === null || _b === void 0 ? void 0 : _b.config;
            const baselineAccessor = baselineComponentConfig === null || baselineComponentConfig === void 0 ? void 0 : baselineComponentConfig.baseline;
            crosshair.accessors = {
                x: (_c = this.components[0]) === null || _c === void 0 ? void 0 : _c.config.x,
                y: flatten(yAccessors),
                yStacked: flatten(yStackedAccessors),
                baseline: baselineAccessor,
            };
            crosshair.g.attr('transform', `translate(${margin.left},${margin.top})`)
                .style('clip-path', `url(#${this._clipPathId})`)
                .style('-webkit-clip-path', `url(#${this._clipPathId})`);
            crosshair.hide();
        }
        (_d = config.annotations) === null || _d === void 0 ? void 0 : _d.g.attr('transform', `translate(${margin.left},${margin.top})`);
        (_e = config.annotations) === null || _e === void 0 ? void 0 : _e.render();
        this._firstRender = false;
    }
    _updateScales(...components) {
        const c = clean(components || this.components);
        this._setScales(...c);
        this._updateScalesDomain(...c);
        this._updateScalesRange(...c);
    }
    _setScales(...components) {
        const { config } = this;
        if (!components)
            return;
        // Set the X and Y scales
        if (config.xScale)
            components.forEach(c => c.setScale(ScaleDimension.X, config.xScale));
        if (config.yScale)
            components.forEach(c => c.setScale(ScaleDimension.Y, config.yScale));
    }
    _updateScalesDomain(...components) {
        const { config } = this;
        if (!components)
            return;
        const componentsWithDomain = components.filter(c => !c.config.excludeFromDomainCalculation);
        // Loop over all the dimensions
        Object.values(ScaleDimension).forEach((dimension) => {
            var _a, _b, _c, _d, _e, _f, _g, _h;
            const [min, max] = extent(merge(componentsWithDomain.map(c => c.getDataExtent(dimension, config.scaleByDomain)))); // Components with undefined dimension accessors will return [undefined, undefined] but d3.extent will take care of that
            const configuredDomain = dimension === ScaleDimension.Y ? config.yDomain : config.xDomain;
            const configuredDomainMinConstraint = dimension === ScaleDimension.Y ? config.yDomainMinConstraint : config.xDomainMinConstraint;
            const configuredDomainMaxConstraint = dimension === ScaleDimension.Y ? config.yDomainMaxConstraint : config.xDomainMaxConstraint;
            const domainMin = (_b = (_a = configuredDomain === null || configuredDomain === void 0 ? void 0 : configuredDomain[0]) !== null && _a !== void 0 ? _a : min) !== null && _b !== void 0 ? _b : 0;
            const domainMax = (_d = (_c = configuredDomain === null || configuredDomain === void 0 ? void 0 : configuredDomain[1]) !== null && _c !== void 0 ? _c : max) !== null && _d !== void 0 ? _d : 1;
            const domain = [
                clamp(domainMin, (_e = configuredDomainMinConstraint === null || configuredDomainMinConstraint === void 0 ? void 0 : configuredDomainMinConstraint[0]) !== null && _e !== void 0 ? _e : Number.NEGATIVE_INFINITY, (_f = configuredDomainMinConstraint === null || configuredDomainMinConstraint === void 0 ? void 0 : configuredDomainMinConstraint[1]) !== null && _f !== void 0 ? _f : Number.POSITIVE_INFINITY),
                clamp(domainMax, (_g = configuredDomainMaxConstraint === null || configuredDomainMaxConstraint === void 0 ? void 0 : configuredDomainMaxConstraint[0]) !== null && _g !== void 0 ? _g : Number.NEGATIVE_INFINITY, (_h = configuredDomainMaxConstraint === null || configuredDomainMaxConstraint === void 0 ? void 0 : configuredDomainMaxConstraint[1]) !== null && _h !== void 0 ? _h : Number.POSITIVE_INFINITY),
            ];
            // Extend the X and Y domains if they're empty and `preventEmptyDomain` was explicitly set to `true`
            // or just the X domain if there is no data provided and `preventEmptyDomain` set to `null`
            if (domain[0] === domain[1]) {
                const hasDataProvided = componentsWithDomain.some(c => { var _a; return ((_a = c.datamodel.data) === null || _a === void 0 ? void 0 : _a.length) > 0; });
                if (config.preventEmptyDomain || (config.preventEmptyDomain === null && (!hasDataProvided || dimension === ScaleDimension.Y))) {
                    domain[1] = domain[0] + 1;
                }
            }
            components.forEach(c => c.setScaleDomain(dimension, domain));
        });
    }
    _updateScalesRange(...components) {
        var _a, _b, _c, _d, _e, _f;
        const { config } = this;
        if (!components)
            return;
        // Set initial scale range
        const isYDirectionSouth = config.yDirection === Direction.South;
        const xRange = [(_a = config.padding.left) !== null && _a !== void 0 ? _a : 0, (_b = this.width - config.padding.right) !== null && _b !== void 0 ? _b : 0];
        const yRange = [(_c = this.height - config.padding.bottom) !== null && _c !== void 0 ? _c : 0, (_d = config.padding.top) !== null && _d !== void 0 ? _d : 0];
        if (isYDirectionSouth)
            yRange.reverse();
        for (const c of components) {
            c.setScaleRange(ScaleDimension.X, (_e = config.xRange) !== null && _e !== void 0 ? _e : xRange);
            c.setScaleRange(ScaleDimension.Y, (_f = config.yRange) !== null && _f !== void 0 ? _f : yRange);
        }
        // Get and combine bleed
        const bleed = components.map(c => c.bleed).reduce((bleed, b) => {
            for (const key of Object.keys(bleed)) {
                const k = key;
                if (bleed[k] < b[k])
                    bleed[k] = b[k];
            }
            return bleed;
        }, { top: 0, bottom: 0, left: 0, right: 0 });
        // Update scale range
        for (const c of components) {
            c.setScaleRange(ScaleDimension.X, [xRange[0] + bleed.left, xRange[1] - bleed.right]);
            c.setScaleRange(ScaleDimension.Y, isYDirectionSouth
                ? [yRange[0] + bleed.top, yRange[1] - bleed.bottom] // if Y axis is directed downwards
                : [yRange[0] - bleed.bottom, yRange[1] + bleed.top] // if Y axis is directed upwards
            );
        }
    }
    _renderAxes(duration) {
        const { config: { xAxis, yAxis } } = this;
        const margin = this._getMargin();
        const axes = clean([xAxis, yAxis]);
        axes.forEach(axis => {
            const offset = axis.getOffset(margin);
            axis.g.attr('transform', `translate(${offset.left},${offset.top})`);
            axis.render(duration);
        });
    }
    _setAutoMargin() {
        const { config: { xAxis, yAxis } } = this;
        // At first we need to set the domain to the scales
        const components = clean([...this.components, xAxis, yAxis]);
        this._updateScalesDomain(...components);
        // Calculate margin required by the axes
        // We do two iterations on the first render, because the amount and size of ticks can change
        //    after new margin are calculated and applied (axes dimensions will change).
        //    That's needed for correct label placement.
        const numIterations = this._firstRender ? 2 : 1;
        for (let i = 0; i < numIterations; i += 1) {
            const axisMargin = { top: 0, bottom: 0, left: 0, right: 0 };
            this._updateScalesRange(...components);
            const axes = clean([xAxis, yAxis]);
            axes.forEach(axis => {
                axis.preRender();
                const m = axis.getRequiredMargin();
                if (axisMargin.top < m.top)
                    axisMargin.top = m.top;
                if (axisMargin.bottom < m.bottom)
                    axisMargin.bottom = m.bottom;
                if (axisMargin.left < m.left)
                    axisMargin.left = m.left;
                if (axisMargin.right < m.right)
                    axisMargin.right = m.right;
            });
            this._axisMargin = axisMargin;
        }
    }
    _getMargin() {
        const { config: { margin } } = this;
        return {
            top: margin.top + this._axisMargin.top,
            bottom: margin.bottom + this._axisMargin.bottom,
            left: margin.left + this._axisMargin.left,
            right: margin.right + this._axisMargin.right,
        };
    }
    destroy() {
        const { components, config: { tooltip, crosshair, annotations, xAxis, yAxis } } = this;
        super.destroy();
        for (const c of components)
            c === null || c === void 0 ? void 0 : c.destroy();
        tooltip === null || tooltip === void 0 ? void 0 : tooltip.destroy();
        crosshair === null || crosshair === void 0 ? void 0 : crosshair.destroy();
        annotations === null || annotations === void 0 ? void 0 : annotations.destroy();
        xAxis === null || xAxis === void 0 ? void 0 : xAxis.destroy();
        yAxis === null || yAxis === void 0 ? void 0 : yAxis.destroy();
    }
}

export { XYContainer };
//# sourceMappingURL=index.js.map
