import { defineStore } from "pinia";
import axios from "axios";
import { useErrorHandling } from "@/modules/core/composables";
import { SummaryChartData, IndicatorDetailsData, NationalOverviewGoals } from "../types";

export const useNationalDataAvailabilityStore = defineStore('national-data-availability', {
    state: () => ({
        isBusy: {
            summary: false,
            overview: false,
            details: false
        },
        country: null,
        goalName: null,
        goalNumber: null,
        region: null,
        summaryChartData: null as SummaryChartData|null,
        overviewData: [] as Array<NationalOverviewGoals>|null,
        indicatorData: [] as Array<IndicatorDetailsData>|null,
    }),
    getters: {
        getImageUrl: (state) => (goalName: string) => {
            const goal = state.overviewData?.find(goal => goal.goal_name === goalName);
            return goal ? goal.image : '';
        },
        formattedOverviewData: (state) => {
            const groupedData = (state.overviewData ?? []).reduce((acc, item) => {
                const { goal_number, goal_name, image, category, count, percentage } = item;
        
                if (!acc[goal_number]) {
                    acc[goal_number] = {
                        goal_number,
                        goal_name,
                        image_url: image,
                        formattedGoalNumber: goal_number < 10 ? `Goal 0${goal_number}` : `Goal ${goal_number}`,
                        dataAvailability: {},
                    };
                }
        
                if (!acc[goal_number].dataAvailability[category]) {
                    acc[goal_number].dataAvailability[category] = {
                        count: 0,
                        percentage: 0,
                    };
                }
        
                acc[goal_number].dataAvailability[category].count += count;
                acc[goal_number].dataAvailability[category].percentage += percentage;
        
                return acc;
            }, {} as Record<string, any>);
        
            return Object.values(groupedData);
        },
        transformedIndicatorData: (state) => {
            return (state.indicatorData ?? []).map(categoryGroup => {
                const transformedData = categoryGroup.data.map(indicator => {
                    const { indicatorNumber, indicatorName, latestYear, compiler, dataSource, sufficientCount, insufficientCount, noDataCount } = indicator;

                    const numberOfAreas = [];

                    if (sufficientCount > 0) {
                        numberOfAreas.push({ category: "Sufficient", count: sufficientCount });
                    }
                    if (insufficientCount > 0) {
                        numberOfAreas.push({ category: "Insufficient", count: insufficientCount });
                    }
                    if (noDataCount > 0) {
                        numberOfAreas.push({ category: "No Data", count: noDataCount });
                    }

                    return {
                        indicatorNumber,
                        indicatorName,
                        latestYear,
                        compiler,
                        dataSource,
                        numberOfAreas
                    };
                });

                return {
                    category: categoryGroup.category,
                    data: transformedData
                };
            });
        },              
        filtredSummaryBreakdown: (state) => {
            return state.summaryChartData?.breakdown.filter(item => item.name !== "Total") || [];
        },        
    },
    actions: {
        async getSummaryChartData (region: string) {
            this.isBusy.summary = true;

            const request = axios.get<SummaryChartData>(
                `/api/data-points/national/data-availability/summary`,
                {
                    params: {
                        region: region
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    summary: false
                },
                summaryChartData: response.data
            })
        },
        async getOverviewData (region: string) {
            this.isBusy.overview = true;

            const request = axios.get<NationalOverviewGoals[]>(
                `/api/data-points/national/data-availability/goals`,
                {
                    params: {
                        region: region
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    overview: false
                },
                overviewData: response.data
            })
        },
        async getIndicatorDetailsData (country: string, goalNumber: string) {
            this.isBusy.details = true;

            const request = axios.get<IndicatorDetailsData[]>(
                `api/data-points/national/data-availability/indicator-details`,
                {
                    params: {
                        country: country,
                        goal_number: goalNumber,
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    details: false
                },
                indicatorData: response.data
            })
        },
        
    }
})