import { defineStore } from "pinia";
import axios from "axios";
import { NationalOverviewData, NationalOverviewDetailData } from "../types";
import { useErrorHandling } from "@/modules/core/composables";

export const useNationalOverviewStore = defineStore('national-overview', {
    state: () => ({
        isBusy: {
            listing: true,
            details: false
        },
        goalId: null,
        region: null,
        overviewGoals: null as NationalOverviewData|null,
        overviewGoalDetails: [] as Array<NationalOverviewDetailData>|null,
    }),
    getters: {  
        getImageUrl: (state) => (goalId: string) => {
            if (!state.overviewGoals) return '';
            const goal = state.overviewGoals.data.find(goal => goal.goal_number === goalId);
            return goal ? goal.image_url : '';
        },
        getFirstTabCategory: (state) => {
            return state.overviewGoalDetails?.[0]?.category || null;
        },
        getFirstIndicators: (state) => {
            if (!state.overviewGoalDetails) return [];
        
            return state.overviewGoalDetails.map(category => {
                const firstIndicator = category.data[0];
                return firstIndicator ? firstIndicator.ind_short_name : '';
            });
        },
        getCategorizedIndicatorData: (state) => {
            return state.overviewGoalDetails?.map((categoryItem) => {
                const indMap: Record<string, any> = categoryItem.data.reduce((acc, item) => {
                    if (!acc[item.ind_number]) {
                        acc[item.ind_number] = {
                            ind_number: item.ind_number,
                            ind_short_name: item.ind_short_name,
                            target_number: item.target_number,
                            area: item.area,
                            trend: item.trend,
                            series_data: {},
                        }
                    }

                    if (!acc[item.ind_number].series_data[item.series]) {
                        acc[item.ind_number].series_data[item.series] = {
                          series: item.series,
                          series_name: item.series_name,
                          series_tag: item.series_tag,
                          unit_name: item.unit_name,
                          series_level: item.series_level,
                          data_points: [],
                        };
                      }
            
                      acc[item.ind_number].series_data[item.series].data_points.push({
                        year: item.year,
                        data_value: item.data_value,
                      });
            
                      return acc;
                }, {} as Record<string, any>);
                
                const data = Object.values(indMap).map((ind) => {
                    ind.series_data = Object.values(ind.series_data);
                    return ind;
                });
        
                return {
                    category: categoryItem.category,
                    data,
                };
            })
        },
    },
    actions: {
        async getOverviewGoals (area: string) {
            this.isBusy.listing = true;

            const request = axios.get<NationalOverviewData>(
                `/api/data-points/national/overview/indicator-percentages`,
                { 
                    params: {
                        area: area
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    listing: false
                },
                overviewGoals: response.data
            })
        },
        async getOverviewGoalDetails (area: string, goal_number: string) {
            this.isBusy.details = true;

            const request = axios.get<NationalOverviewDetailData[]>(
                `/api/data-points/national/overview/indicator-trends`,
                { 
                    params: {
                        area: area,
                        goal_number: goal_number
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    details: false
                },
                overviewGoalDetails: response.data
            })
        }
    }
})