import { defineStore } from "pinia";
import axios from "axios";
import { FormattedPrioritiesTargetData, GoalData, NationalPrioritiesData, NationalPrioritiesTargetData } from "../types";
import { useErrorHandling } from "@/modules/core/composables";

export const useNationalPrioritiesStore = defineStore('national-priorities', {
    state: () => ({
        isBusy: true,
        goalId: null,
        region: null,
        sdgGoals: [] as GoalData[],
        sdgPrioritiesChart: [] as NationalPrioritiesData[],
        targetId: '',
        targetChart: null as FormattedPrioritiesTargetData|null
    }),
    getters: {
        getImageUrl: (state) => (goalId: string) => {
            const goal = state.sdgGoals?.find(goal => goal.goal_number === goalId);
            return goal ? goal.image_url : '';
        },
        getFirstChartCategory: (state) => {
            return state.sdgPrioritiesChart?.[0]?.category_name|| null;
        },
        getChartData(state) {
            return state.sdgPrioritiesChart; 
        },
        categorizedPrioritiesData: (state) => {
            return state.sdgPrioritiesChart?.map((categoryItem) => {
                const groupedData = categoryItem.data.reduce((acc, item) => {
                    const key = `${item.ind_number}_${item.series}`;
                    if (!acc[key]) {
                        acc[key] = {
                            sdg_performers_data_id: item.sdg_performers_data_id,
                            ind_number: item.ind_number,
                            ind_short_name: item.ind_short_name || "NA",
                            series_name: item.series_name || "NA",
                            series: item.series || "NA",
                            area: item.area || "NA",
                            unit_name: item.unit_name || "NA",
                            series_level: item.series_level || "NA",
                            trend: item.trend || "No Data",
                            data_points: [],
                        };
                    }
                    acc[key].data_points.push({
                        year: item.year || "NA",
                        data_value: parseFloat(item.data_value) || 0,
                    });
        
                    return acc;
                }, {} as Record<string, any>);
        
                const transformedData = Object.values(groupedData);
        
                return {
                    category: categoryItem.category_name,
                    data: transformedData,
                };
            });
        },        
    },
    actions: {
        async getSdgGoals () {
            this.isBusy = true;

            const request = axios.get<GoalData[]>(
                `/api/goals`
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: false,
                sdgGoals: response.data
            })
        },
        async getPrioritiesData (goal_number: string, area: string) {
            this.isBusy = true;

            const request = axios.get(
                `/api/data-points/national/priorities`,
                { 
                    params: {
                        area: area,
                        goal_number: goal_number
                    }
                }
            )
            
            const response = await useErrorHandling(request);
            
            this.$patch({
                isBusy: false,
                sdgPrioritiesChart: response.data,
                targetId: response.data[0]?.data[0].sdg_performers_data_id,
                targetChart: response.data?.flatMap((category: { data: NationalPrioritiesTargetData; }) => category.data).find((data: { sdg_performers_data_id: string; }) => data.sdg_performers_data_id === this.targetId) || null
            })
        },
        async getNewTargetData () {
            const categoryData = this.categorizedPrioritiesData.find(category => 
                category.data.some(item => item.sdg_performers_data_id === this.targetId)
            );
            
            if (categoryData) {
                this.targetChart = categoryData.data.find(item => item.sdg_performers_data_id === this.targetId) || null;
            } else {
                this.targetChart = null;
            }
        }
    }
})