import { defineStore } from "pinia";
import axios from "axios";
import { GoalDetailsData, GoalsData } from "../types";
import { useErrorHandling } from "@/modules/core/composables";

export const useRegionalDashboardStore = defineStore('regional-dashboard', {
    state: () => ({
        isBusy: {
            listing: true,
            details: false
        },
        goalId: null,
        region: null,
        sdgGoals: [] as Array<GoalsData>|null,
        goalDetails: [] as Array<GoalDetailsData>|null,
    }),
    getters: {
        getImageUrl: (state) => (goalNumber: string) => {
            const goal = state.sdgGoals?.find(g => g.goal_number === goalNumber);
            return goal?.image_url || '';
        },
        categorizedGoals: (state) => {    
            return state.sdgGoals?.reduce((result, goal) => {
                const { goal_number, goal_name, image_url, target_number, target_color, target_long_name } = goal;
    
                if (!result[goal_number]) {
                    result[goal_number] = {
                        goal_name,
                        goal_number,
                        image_url,
                        targets: []
                    };
                }
    
                result[goal_number].targets.push({
                    target_number,
                    target_color,
                    target_long_name
                });
    
                return result;
            }, {} as Record<string, {
                goal_name: string;
                goal_number: string;
                image_url: string | null;
                targets: Array<{
                    target_number: string;
                    target_color: string;
                    target_long_name: string;
                }>;
            }>);
        }
    },
    actions: {
        async getSdgGoals (region: string) {
            this.isBusy.listing = true;

            const request = axios.get<GoalsData[]>(
                `/api/data-points/regional/dashboard`,
                {
                    params: {
                        region: region
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    listing: false
                },
                sdgGoals: response.data
            })
        },
        async getGoalDetails (goal_number: string, region: string) {
            this.isBusy.details = true;

            const request = axios.get<GoalDetailsData[]>(
                `/api/data-points/regional/dashboard-detail-chart`,
                { 
                    params: {
                        goal_number: goal_number,
                        region: region
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    details: false
                },
                goalDetails: response.data
            })
        }
    }
})