import { defineStore } from "pinia";
import axios from "axios";
import { useErrorHandling } from "@/modules/core/composables";
import { RegionalAvailabilityData, RegionalOverviewGoalsData, RegionalIndicatorData, RegionalCountryTerritoriesData } from "../types";

export const useRegionalDataAvailabilityStore = defineStore('regional-data-availability', {
    state: () => ({
        isBusy: {
            availability: false,
            overview: false,
            indicator: false,
            countriesAndTerritories: false
        },
        country: null,
        goalName: null,
        goalNumber: null,
        region: null,
        availabilityData: [] as Array<RegionalAvailabilityData> | null,
        overviewData: [] as Array<RegionalOverviewGoalsData> | null,
        indicatorData: [] as Array<RegionalIndicatorData> | null,
        countryTerritoriesData: [] as Array<RegionalCountryTerritoriesData> | null,
    }),
    getters: {
        getImageUrl: (state) => (goalName: string) => {
            const goal = state.overviewData?.find(goal => goal.formattedGoalNumber === goalName);
            return goal ? goal.image_url : '';
        },
        sortedSufficientCountryTerritoriesData: (state) => {
            return state.countryTerritoriesData ? state.countryTerritoriesData.sort((a, b) => b.Sufficient - a.Sufficient) : [];
        },
        sortedInsufficientCountryTerritoriesData: (state) => {
            return state.countryTerritoriesData ? state.countryTerritoriesData.sort((a, b) => b["Insufficient Data"] - a["Insufficient Data"]) : [];
        },
        transformedIndicatorDetails: (state) => {
            return (state.indicatorData ?? []).map(detailGroup => {
                const { indicatorNumber, indicatorName, sufficientCount, insufficientCount, noDataCount } = detailGroup;

                const numberOfAreas = [];

                if (sufficientCount > 0) {
                    numberOfAreas.push({ category: "Sufficient", count: sufficientCount });
                }
                if (insufficientCount > 0) {
                    numberOfAreas.push({ category: "Insufficient", count: insufficientCount });
                }
                if (noDataCount > 0) {
                    numberOfAreas.push({ category: "No Data", count: noDataCount });
                }

                return {
                    indicatorNumber,
                    indicatorName,
                    numberOfAreas
                };
            })
        }
    },
    actions: {
        async getAvailabilityChartData(region: string) {
            this.isBusy.availability = true;

            const request = axios.get<RegionalAvailabilityData[]>(
                `/api/data-points/regional/data-availability/years`,
                {
                    params: {
                        region: region
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    availability: false
                },
                availabilityData: response.data
            })
        },
        async getOverviewGoals(region: string) {
            this.isBusy.overview = true;

            const request = axios.get<RegionalOverviewGoalsData[]>(
                `/api/data-points/regional/data-availability/goals`,
                {
                    params: {
                        region: region
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    overview: false
                },
                overviewData: response.data.map(goal => ({
                    ...goal,
                    formattedGoalNumber: goal.goal_number < 10 ? `Goal 0${goal.goal_number}` : `Goal ${goal.goal_number}`
                }))
            })
        },
        async getIndicatorDetailsData(region: string, goalNumber: string) {
            this.isBusy.indicator = true;

            const request = axios.get<RegionalIndicatorData[]>(
                `api/data-points/regional/data-availability/indicator-details`,
                {
                    params: {
                        region: region,
                        goal_number: goalNumber,
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    indicator: false
                },
                indicatorData: response.data
            })
        },
        async getCountryTerritoriesData(region: string) {
            this.isBusy.countriesAndTerritories = true;

            const request = axios.get<RegionalCountryTerritoriesData[]>(
                `api/data-points/regional/data-availability/countries`,
                {
                    params: {
                        region: region,
                    }
                }
            )

            const response = await useErrorHandling(request);

            this.$patch({
                isBusy: {
                    ...this.isBusy,
                    countriesAndTerritories: false
                },
                countryTerritoriesData: response.data
            })
        },

    }
})